<?php


namespace App\Service\Students\Grade;


use App\Entity\Classes;
use App\Entity\ClassGrade;
use App\Entity\ClassGradeTemplate;
use App\Entity\ClassTemplate;
use App\Entity\GradeCategories;
use App\Entity\StudentGrades;
use App\Repository\ClassGradeRepository;
use App\Repository\ClassGradeTemplateRepository;
use App\Repository\GradeCategoriesRepository;
use App\Repository\StudentGradesRepository;
use App\Service\DefaultFunction;
use App\Service\StudentClasses;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\ParameterBag;
use Symfony\Component\Validator\ConstraintViolationListInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;

class ClassGradeService extends ClassGradeTemplateService
{

    /**
     * @var DefaultFunction
     */
    private $defaultFunction;
    /**
     * @var EntityManagerInterface
     */
    private $entityManager;
    /**
     * @var ValidatorInterface
     */
    private $validator;
    /**
     * @var ClassGradeRepository
     */
    private $classGradeRepository;
    /**
     * @var StudentClasses
     */
    private $studentClasses;
    /**
     * @var GradeCategoriesRepository
     */
    private $gradeCategoriesRepository;
    /**
     * @var StudentGradesRepository
     */
    private $studentGradesRepository;
    /**
     * @var ClassGradeTemplateRepository
     */
    private $classGradeTemplateRepository;

    public function __construct(
        GradeCategoriesRepository $gradeCategoriesRepository,
        EntityManagerInterface $entityManager,
        DefaultFunction $defaultFunction,
        StudentGradesRepository $studentGradesRepository,
        ValidatorInterface $validator,
        StudentClasses $studentClasses,
        ClassGradeTemplateRepository $classGradeTemplateRepository,
        ClassGradeRepository $classGradeRepository
    ) {

        parent::__construct(
            $gradeCategoriesRepository,
            $entityManager,
            $defaultFunction,
            $studentGradesRepository,
            $validator,
            $studentClasses,
            $classGradeTemplateRepository,
            $classGradeRepository
        );

        $this->defaultFunction = $defaultFunction;
        $this->entityManager = $entityManager;
        $this->validator = $validator;
        $this->classGradeRepository = $classGradeRepository;
        $this->studentClasses = $studentClasses;
        $this->gradeCategoriesRepository = $gradeCategoriesRepository;
        $this->studentGradesRepository = $studentGradesRepository;
        $this->classGradeTemplateRepository = $classGradeTemplateRepository;
    }

    /**
     * @param ClassTemplate $classTemplate
     * @param Classes $classes
     * @return string
     * Purpose: Save data to ClassGrade by Copying from ClassGradeTemplate
     */
    public function copyStudentClassGradeTemplateToClassGrade(ClassTemplate $classTemplate, Classes $classes)
    {
        $response = [];
        if ($classTemplate instanceof ClassTemplate) {
            $listOfClassGradeTemplate = $classTemplate->getClassGradeTemplates()->toArray();
            /** @var ClassGradeTemplate $value */
            foreach ($listOfClassGradeTemplate as $key => $value) {
                $bag = new ParameterBag();
                $bag->set('gradeCat', $value->getStudentGradeCategory()->getId());
                $bag->set('classGradeTemplateInstance', $value);
                $bag->set('classInstance', $classes);
                $bag->set('gradeDesc', $value->getGradeDesc());
                $bag->set('gradeNumberEqualOrGreater', $value->getMarksEqualOrGreaterThan());
                $bag->set('gradeName', $value->getGradeName());
                $response = $this->defaultFunction->push_error($response, $this->saveClassGrades($bag));
            }
        }

        return $this->defaultFunction->purify_success_error_response_in_bulk_operation($response);
    }


    /**
     * @param ParameterBag $bag
     * @return array|mixed|string|ConstraintViolationListInterface
     * Purpose: Save data inside Class Grade table
     */
    public function saveClassGrades(ParameterBag $bag)
    {
        $classGrade = $this->getClassGrade($bag->get('gradeItemRef'));
        if (!$classGrade instanceof ClassGrade) {
            $classGrade = new ClassGrade();
        }

        /*
         * ================ Begin:: Grade Category
         * */

        ## get grade category
        ## 1. check is grade instance is given.
        if ($bag->get('gradeCategoryInstance') instanceof GradeCategories) {
            $gradeCategory = $bag->get('gradeCategoryInstance');
        } else {
            $gradeCategory = $this->getGradeCategory($bag->get('gradeCat'));
        }
        ## If grade Category has found then save.
        if ($gradeCategory instanceof GradeCategories) {
            ## set grade category reference
            $classGrade->setStudentGradeCategory($gradeCategory);
        }

        /*
         * ================ End:: Grade Category
         * */


        /*
         * ================ Begin:: ClassGradeTemplate
         * */

        ## set the ClassGradeTemplate Ref
        if ($bag->get('classGradeTemplateInstance') instanceof ClassGradeTemplate) {
            $classGradeTemplate = $bag->get('classGradeTemplateInstance');
        } else {
            $classGradeTemplate = $this->getClassGradeTemplate($bag->get('classGradeTemplate'));
        }

        ## If class template has found then save.
        if ($classGradeTemplate instanceof ClassGradeTemplate) {
            ## set class template reference
            $classGrade->setClassGradeTemplate($classGradeTemplate);
        }

        /*
         * ================ End:: ClassGradeTemplate
         * */


        /*
         * ================ Begin:: Class
         * */

        ## set the Class Reference
        if ($bag->get('classInstance') instanceof Classes) {
            $Class = $bag->get('classInstance');
        } else {
            $Class = $this->studentClasses->get_the_class($bag->get('class'));
        }

        ## set the Class reference
        if ($Class instanceof Classes) {
            $classGrade->setClass($Class);
        }

        /*
         * ================ End:: Class
         * */

        ## set grade description
        $classGrade->setClassGradeDesc($bag->get('gradeDesc', ''));
        ## grade applicable when marks are equal or greater than
        $classGrade->setClassMarksEqualOrGreaterThan($bag->get('gradeNumberEqualOrGreater', 0));
        ## set grade name
        $classGrade->setClassGradeName($bag->get('gradeName', ''));

        $errors = $this->validator->validate($classGrade);
        $errors = $this->defaultFunction->parseConstraintViolationInString($errors);

        if (empty($errors)) {

            ## Duplication Check
            $duplicationExists = $this->classGradeRepository->gradeCategoryGradeNameGradeMarksDuplicationCheck(
                $Class->getId(),
                $bag->get('gradeName'),
                $bag->get('gradeNumberEqualOrGreater')
            );

            if ($duplicationExists) {
                ## if records are duplication
//                $errors = 'Trying to save duplicate record';
                $errors = 'OK';
            } else {
                ## if records are not duplication then save
                try {
                    $this->entityManager->persist($classGrade);
                    $this->entityManager->flush();
                    $errors = 'OK';
                } catch (\Exception $exception) {
                    $errors = $exception->getMessage();
                }
            }
        }

        return $errors;
    }


    /**
     * @param ParameterBag $bag
     * @return array|mixed
     * Purpose: Save Bulk ClassGrades - this method filter data and provide to require method.
     */
    public function saveBulkGradeIntoClass(ParameterBag $bag)
    {
        $response = [];
        if (!empty($bag->get('gradeName')) && array($bag->get('gradeName'))) {
            for ($x = 0; $x < count($bag->get('gradeName')); $x++) {
                $bag2 = new ParameterBag();

                ## grade reference if want to edit
                $bag2->set(
                    'gradeItemRef',
                    !empty($bag->get('gradeItemRef')[$x]) ? $bag->get('gradeItemRef')[$x] : null
                );
                ## grade name
                $bag2->set('gradeName', !empty($bag->get('gradeName')[$x]) ? $bag->get('gradeName')[$x] : '');
                ## grade number if equal or greater than
                $bag2->set(
                    'gradeNumberEqualOrGreater',
                    !empty($bag->get('gradeNumberEqualOrGreater')[$x]) ? $bag->get(
                        'gradeNumberEqualOrGreater'
                    )[$x] : 0
                );
                ## grade description
                $bag2->set('gradeDesc', !empty($bag->get('gradeDesc')[$x]) ? $bag->get('gradeDesc')[$x] : '');
                ## grade category
                $bag2->set('gradeCat', $bag->get('gradeCat'));
                ## class Grade template id
                $bag2->set(
                    'classGradeTemplate',
                    !empty($bag->get('classGradeTemplate')[$x]) ? $bag->get('classGradeTemplate')[$x] : null
                );
                ## class id
                $bag2->set('class', $bag->get('class'));
                ## save grades.
                $response = $this->defaultFunction->push_error($response, $this->saveClassGrades($bag2));
            }
        }

        ## get the error or success response from bulk action
        return $this->defaultFunction->purify_success_error_response_in_bulk_operation($response);
    }


    /**
     * @param null $classGrade
     * @param null $key
     * @param null $value
     * @return ClassTemplate[]|mixed|object|string|null
     * Purpose Get the Class Grade Template
     */
    public
    function getClassGrade(
        $classGrade = null,
        $key = null,
        $value = null
    ) {
        $response = null;
        try {
            if (!empty($classGrade)) {
                $response = $this->classGradeRepository->find($classGrade);
            } else {
                if (!empty($key) && !empty($value)) {
                    $response = $this->classGradeRepository->findBy([$key => $value]);
                }
            }
        } catch (\Exception $exception) {
            $response = $exception->getMessage();
        }

        return $response;
    }


    /**
     * @param $classID
     * @param $marksPercentage
     * @return mixed|string|null
     * Purpose Grade Calculator
     */
    public function calculateGradeFromMarksPercentage($classID, $marksPercentage)
    {
        ## get the class grade
        $listOfClassGrade = $this->getClassGrade(null, 'class', $classID);
        /** @var ClassGrade $value */
        $MarksList = [];
        foreach ($listOfClassGrade as $key => $value) {
            $MarksList[$value->getClassMarksEqualOrGreaterThan()] = $value;
        }

        $MarksPercentage = array_keys($MarksList);

        ## sort array in desc order
        rsort($MarksPercentage);

        $grade = null;
        foreach ($MarksPercentage as $key => $value) {
            if ($value == $marksPercentage || $marksPercentage > $value) {
                $grade = $MarksList[$value];
                break;
            }
        }

        if ($grade instanceof ClassGrade) {
            $grade = $grade->getClassGradeName();
        } else {
            $grade = 'F';
        }

        return $grade;
    }

    /**
     * @param ParameterBag|null $bag
     * @return ClassGrade
     * Purpose: Get the list of all grades with attached to the Class
     */
    public function listOfClassGrade(ParameterBag $bag = null)
    {
        return $this->classGradeRepository->getGradesList($bag);
    }
}
