<?php

namespace App\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * @ORM\Entity(repositoryClass="App\Repository\ClassGradeTemplateRepository")
 */
class ClassGradeTemplate
{
    /**
     * @ORM\Id()
     * @ORM\GeneratedValue()
     * @ORM\Column(type="integer")
     */
    private $id;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\GradeCategories", inversedBy="classGradeTemplates")
     * @ORM\JoinColumn(nullable=false)
     * @Assert\NotBlank(message="Grade category reference should not be null")
     * @Assert\Valid
     */
    private $student_grade_category;

    /**
     * @ORM\Column(type="string", length=5)
     * @Assert\NotBlank(message="Grade name should not be null")
     * @Assert\Length(min=1, max=5, minMessage="Grade name must be at least {{ limit }} characters long", maxMessage =
     *                       "Grade name cannot be longer than {{ limit }} characters")
     */
    private $grade_name;

    /**
     * @ORM\Column(type="string", length=255, nullable=true)
     * @Assert\Length(max=255,  maxMessage =
     *                       "Grade Description be longer than {{ limit }} characters")
     */
    private $grade_desc;

    /**
     * @ORM\Column(type="smallint")
     * @Assert\NotBlank
     * @Assert\Length(min=1, max=6, minMessage="Marks must be at least {{ limit }} characters long", maxMessage =
     *                       "Marks cannot be longer than {{ limit }} characters")
     */
    private $marks_equal_or_greater_than;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\ClassTemplate", inversedBy="classGradeTemplates")
     * @ORM\JoinColumn(nullable=false)
     * @Assert\NotBlank(message="Class template reference should not be null")
     * @Assert\Valid
     */
    private $class_template;

    /**
     * @ORM\OneToMany(targetEntity="App\Entity\ClassGrade", mappedBy="class_grade_template")
     */
    private $classGrades;

    public function __construct()
    {
        $this->classGrades = new ArrayCollection();
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getStudentGradeCategory(): ?GradeCategories
    {
        return $this->student_grade_category;
    }

    public function setStudentGradeCategory(?GradeCategories $student_grade_category): self
    {
        $this->student_grade_category = $student_grade_category;

        return $this;
    }

    public function getGradeName(): ?string
    {
        return $this->grade_name;
    }

    public function setGradeName(string $grade_name): self
    {
        $this->grade_name = $grade_name;

        return $this;
    }

    public function getGradeDesc(): ?string
    {
        return $this->grade_desc;
    }

    public function setGradeDesc(?string $grade_desc): self
    {
        $this->grade_desc = $grade_desc;

        return $this;
    }

    public function getMarksEqualOrGreaterThan(): ?int
    {
        return $this->marks_equal_or_greater_than;
    }

    public function setMarksEqualOrGreaterThan(int $marks_equal_or_greater_than): self
    {
        $this->marks_equal_or_greater_than = $marks_equal_or_greater_than;

        return $this;
    }

    public function getClassTemplate(): ?ClassTemplate
    {
        return $this->class_template;
    }

    public function setClassTemplate(?ClassTemplate $class_template): self
    {
        $this->class_template = $class_template;

        return $this;
    }

    /**
     * @return Collection|ClassGrade[]
     */
    public function getClassGrades(): Collection
    {
        return $this->classGrades;
    }

    public function addClassGrade(ClassGrade $classGrade): self
    {
        if (!$this->classGrades->contains($classGrade)) {
            $this->classGrades[] = $classGrade;
            $classGrade->setClassGradeTemplate($this);
        }

        return $this;
    }

    public function removeClassGrade(ClassGrade $classGrade): self
    {
        if ($this->classGrades->contains($classGrade)) {
            $this->classGrades->removeElement($classGrade);
            // set the owning side to null (unless already changed)
            if ($classGrade->getClassGradeTemplate() === $this) {
                $classGrade->setClassGradeTemplate(null);
            }
        }

        return $this;
    }
}
