<?php


namespace App\Service;


use Gedmo\Exception\UploadableException;
use http\Env;
use Psr\Container\ContainerInterface;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Finder\Finder;
use Symfony\Component\HttpFoundation\File\Exception\FileException;
use Symfony\Component\HttpFoundation\ParameterBag;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;

class FileManagment
{

    /**
     * @var ParameterBag
     */
    private $parameter_bag;
    /**
     * @var Filesystem
     */
    private $filesystem;
    /**
     * @var Finder
     */
    private $finder;
    /**
     * @var RequestStack
     */
    private $request_stack;
    /**
     * @var UrlGeneratorInterface
     */
    private $url_generator;
    /**
     * @var ContainerInterface
     */
    private $container;

    /**
     * FileManagment constructor.
     *
     * @param Filesystem $filesystem
     * @param ParameterBagInterface $parameter_bag
     * @param RequestStack $request_stack
     * @param UrlGeneratorInterface $url_generator
     */
    public function __construct(
        Filesystem $filesystem,
        ParameterBagInterface $parameter_bag,
        RequestStack $request_stack,
        UrlGeneratorInterface $url_generator
    ) {

        $this->parameter_bag = $parameter_bag;
        $this->filesystem = $filesystem;
        $this->request_stack = $request_stack;
        $this->url_generator = $url_generator;
    }

    /**
     * @param array $args
     *  File array | FileParameterBag | $fileArray.
     *  Allowed Extensions | Array | $allowedExtension
     *  Upload public or Privately | boolean   | $public
     *  Path| string  | path
     *  Return Uploaded file path or success message | string | returnPath
     *  Delete folder if exits | boolean | deleteFolder
     *
     * @return array
     *
     */
    public function uploadFile($args = [])
    {

        ## response of the action
        $response = [
            'extra_data' => '',
            'response' => '',
        ];

        ## FIle array.
        if (empty($args['fileArray'])) {
            $response['response'] = 'File array is empty';
        }

        ## allowed extensions check.
        ## if allowed extension variable is not empty and file extension is not exits in the given array then back sent him back with the message.
        if (!empty($args['allowedExtension']) && !in_array(
                $args['fileArray']->guessExtension(),
                $args['allowedExtension']
            )) {
            $response['response'] = 'This file extension is not allowed, Allowed extension are '.implode(
                    ',',
                    $args['allowedExtension']
                );
        }

        ## return back if we've errors.
        if (!empty($response['response'])) {
            return $response;
        }

        ## get the file name.
        //	$fileName = $args['fileArray']->getClientOriginalName() . $args['fileArray']->getExtension();
        $fileName = $args['fileArray']->getClientOriginalName();


        ## Upload in private or public directory.
        $args['public'] = empty($args['public']) ? true : $args['public'];
        $basePath = $this->get_base_path($args['public']);

        ## make directory if not exits.
        self::make_directory($basePath);

        if ($args['path']) {
            $basePath .= $args['path'].DIRECTORY_SEPARATOR;
            ## make directory if not exits.
            self::make_directory($basePath);
        }

        ## delete the file if exits.
        if (!empty($args['deleteFolder']) && $args['deleteFolder'] === true) {
            $deleteFile_ifExitsArray = [
                'public' => $args['public'],
                'path' => $args['path'],
            ];
            $this->deleteFolder($deleteFile_ifExitsArray);
        }


        ## upload files
        try {
            $args['fileArray']->move($basePath, $fileName);
        } catch (FileException $e) {
            throw  new UploadableException('File not uploaded');
        }

        ## this parameter will decided what to return, Return path or the success message.
        if (!empty($args['returnPath'])) {
            $response['extra_data'] = $basePath.$fileName;
            $response['response'] = 'OK';
            ## return uploaded file url
            $response['uploadedFileUrl'] = $this->convertBasePathIntoUrl($basePath).$fileName;
        } else {
            $response['extra_data'] = $args['path'];
            $response['response'] = 'OK';
            $response['uploadedFileUrl'] = $this->convertBasePathIntoUrl($args['path']).$fileName;
        }

        ## return response.
        return $response;

    }

    /**
     * Make directory if not exits.
     *
     * @param $path
     */
    public function make_directory($path)
    {
        ## if file not exits then upload it.
        if (!$this->filesystem->exists($path)) {
            $this->filesystem->mkdir($path);
        }
    }

    /**
     * @param array $args
     *  base folder is public or private | bool | public
     *  path of the folder | string | path
     *  What you need a file url or the path of the file | bool | need_url
     *
     * @param null $basePathGiven
     *
     * @return array
     */
    public function getUploadedFile($args = [])
    {
        $fileList = [];

        ## Upload in private or public directory.
        if (isset($args['public']) && $args['public']) {
            $basePath = $this->parameter_bag->get('publicUploadDir');
        } else {
            if (isset($args['projectPath']) && $args['projectPath']) {
                $basePath = $this->parameter_bag->get('RootPath');
            } else {
                $basePath = $this->parameter_bag->get('privateUploadDir');
            }
        }

        ## if path is provide the get that.
        if (!empty($args['path'])) {
            $basePath = $basePath.$args['path'].DIRECTORY_SEPARATOR;
        }

        ## if absolute path is given then don't need to add the base path.
        if (!empty($args['absPath'])) {
            $basePath = $args['absPath'];
        }

        ## check given folder is exits or not
        if ($this->filesystem->exists($basePath)) {
            ## symfony finder component
            $finder = new Finder();


            ## get files from folder.
            if (isset($args['findInDirectory']) && $args['findInDirectory']) {
                ## search for the folders
                $finder->directories()->in($basePath);
            } else {
                ## search for the files
                $finder->files()->in($basePath);
            }

            ## get all these matches files.
            if (isset($args['finderPattern']) && $args['finderPattern']) {
                $finder->name($args['finderPattern']);
            }

            ## get the file order.
            if (!empty($args['finderOrder'])) {
                if ($args['finderOrder'] == 'accessTime') {
                    $finder->sortByAccessedTime();
                } else {
                    if ($args['finderOrder'] == 'changeTime') {
                        $finder->sortByChangedTime();
                    } else {
                        if ($args['finderOrder'] == 'modifiedTime') {
                            $finder->sortByModifiedTime();
                        }
                    }
                }
            }


            ## if finder has result.
            if ($finder->hasResults()) {
                foreach ($finder as $key => $file) {
                    ## if user need a url instead of path.
                    if (!empty($args['need_url']) && $args['need_url'] || !empty($args['need_only_file_relative_path'])) {
                        array_push($fileList, $this->convertBasePathIntoUrl($basePath.$file->getFilename()));
                    } else {
                        if (!empty($args['need_only_file_or_folder_name'])) {
                            ## return file or folder name only
                            array_push($fileList, $file->getFilename());
                        } else {
                            if (!empty($args['getFileContent'])) {
                                ## get file content
                                array_push($fileList, $file->getContents());
                            } else {
                                ## get the file with base name
                                array_push($fileList, $basePath.DIRECTORY_SEPARATOR.$file->getFilename());
                            }
                        }
                    }
                }
            }
        }

        return $fileList;
    }

    /**
     * delete file if exits.
     *
     * @param array $args
     */
    public function deleteFile($args = [])
    {
        $filePath = $this->mergeBasePath($args['public'], $args['path']);

        ## check file existence.
        if ($this->filesystem->exists($filePath)) {
            $this->filesystem->remove($filePath);
        }

    }

    /**
     * Get base path of file upload.
     *
     * @param bool $public
     *
     * @return mixed
     */
    public function get_base_path(bool $public)
    {
        $basePath = $this->parameter_bag->get('privateUploadDir');
        ## Upload in private or public directory.
        if ($public) {
            $basePath = $this->parameter_bag->get('publicUploadDir');
        }

        return $basePath;
    }

    /**
     * @param $fileRelativePath
     *
     * @return string
     * PURPOSE: Get absolute url of the file, it always send public folder path
     */
    public function getAbsolutePathOfFile($fileRelativePath = null)
    {
        $path = $this->parameter_bag->get('kernel.project_dir');
        if (!empty($path)) {
            $path = $path.DIRECTORY_SEPARATOR.$fileRelativePath;
        }

        return $path;
    }

    /**
     * delete folder if exits.
     *
     * @param array $args
     */
    public function deleteFolder($args = [])
    {
        $folderPath = $this->mergeBasePath($args['public'], $args['path']);
        if ($this->filesystem->exists($folderPath)) {
            $this->filesystem->remove($folderPath);
        }
    }

    /**
     * merge base path with the given path
     *
     * @param $publicPath
     * @param $extendedPath
     *
     * @return string
     */
    public function mergeBasePath($publicPath, $extendedPath)
    {
        $basePath = $this->get_base_path($publicPath);

        return $basePath.$extendedPath;
    }

    /**
     * Convert base path into the url
     *
     * @param $basePath
     *
     * @return string
     */
    public function convertBasePathIntoUrl($basePath)
    {
        $replace = [
            $this->parameter_bag->get('kernel.project_dir').'/' => '',
            '\\' => '/',
        ];

        return strtr($basePath, $replace);
    }


}
