<?php


namespace App\Service\Fees;


use App\Entity\Classes;
use App\Entity\ClassFee;
use App\Entity\ClassFeeSchedule;
use App\Entity\ClassFeeTemplate;
use App\Entity\ClassSubjectTemplate;
use App\Entity\ClassTemplate;
use App\Entity\StudentFee;
use App\Repository\ClassFeeRepository;
use App\Repository\ClassFeeTemplateRepository;
use App\Repository\StudentFeeRepository;
use App\Service\DefaultFunction;
use Doctrine\DBAL\Exception\ForeignKeyConstraintViolationException;
use Doctrine\ORM\EntityManagerInterface;
use Gedmo\Exception\ReferenceIntegrityStrictException;
use Symfony\Component\HttpFoundation\ParameterBag;

class StudentFeeService
{

    private $errorArray = [];
    /**
     * @var DefaultFunction
     */
    private $default_function;
    /**
     * @var StudentFeeRepository
     */
    private $student_fee_repository;
    /**
     * @var EntityManagerInterface
     */
    private $entity_manager;
    /**
     * @var ClassFeeTemplateRepository
     */
    private $class_fee_template_repository;
    /**
     * @var ClassFeeRepository
     */
    private $class_fee_repository;

    public function __construct(
        DefaultFunction $default_function,
        StudentFeeRepository $student_fee_repository,
        EntityManagerInterface $entity_manager,
        ClassFeeTemplateRepository $class_fee_template_repository,
        ClassFeeRepository $class_fee_repository
    ) {
        $this->default_function = $default_function;
        $this->student_fee_repository = $student_fee_repository;
        $this->entity_manager = $entity_manager;
        $this->class_fee_template_repository = $class_fee_template_repository;
        $this->class_fee_repository = $class_fee_repository;
    }

    /**
     * Student Fee
     *
     * */


    ## fee validation, empty checks.
    public function validateAddEditFee(ParameterBag $bag)
    {
        $response = [];

        ## fee name should not be empty
        if (empty($bag->get('name'))) {
            $response = $this->default_function->push_error($response, 'Fee name should not be empty');
        }

        ## fee amount should not be empty or less than 7 characters.
        if (empty($bag->get('amount')) || strlen($bag->get('amount')) > 7) {
            $response = $this->default_function->push_error(
                $response,
                'Fee name should not be empty or should be smaller then 7 digits'
            );
        }

        ## When fees will be charged
        //		if ( empty( $bag->get( 'charge_atAd' ) ) ) {
        //			$response = $this->default_function->push_error( $response, ' When fee will be charged, at admission time or custom time' );
        //		}

        return $response;
    }

    public function addEditfee(\Symfony\Component\HttpFoundation\ParameterBag $request)
    {
        ## validation check
        if (!empty($validationResponse = $this->validateAddEditFee($request))) {
            return $validationResponse;
        }

        $FeeEntity = null;

        ## decide what to want, add or edit?
        if (!empty($request->get('___rand__f'))) {
            ## edit fee
            $FeeEntity = $this->getFeeFromStudentFeeTable($request->get('___rand__f'));
        }

        ## if instance not found then create a new instance.
        if (!$FeeEntity instanceof StudentFee) {
            $FeeEntity = new StudentFee();
        }

        ## duplication check.
        $bagDuplication = new ParameterBag();
        $bagDuplication->set('name', trim($request->get('name')));
        $bagDuplication->set('whereNotIn', $FeeEntity->getId());
        $duplicaitonResponse = $this->student_fee_repository->duplicationCheck($bagDuplication);
        if (!empty($duplicaitonResponse)) {
            $this->errorArray = $this->default_function->push_error(
                $this->errorArray,
                $request->get('name').' has already exist with same name'
            );
        }

        if (!empty($this->errorArray)) {
            return $this->errorArray;
        }


        try {
            $FeeEntity->setName(trim($request->get('name')));
            $FeeEntity->setDescription($request->get('fee_details'));
            $FeeEntity->setAmount((float)$request->get('amount'));
            $FeeEntity->setRecurring($request->get('recurring'));
            $FeeEntity->setChargetAt((int)$request->get('charge_at'));
            $FeeEntity->setIsChargedAtAdmissionTime(
                $this->default_function->parse__boolean($request->get('charge_atAd'))
            );

            $this->entity_manager->persist($FeeEntity);
            $this->entity_manager->flush();

        } catch (\Exception $exception) {
            $this->errorArray = $this->default_function->push_error($this->errorArray, $exception->getMessage());
        }


        ## return if we have an error
        if (!empty($this->errorArray)) {
            return $this->errorArray;
        } else {
            return 'OK';
        }
    }

    ## get fee from the student fee table
    public function getFeeFromStudentFeeTable(int $FeeID = null, $column = null, $value = null)
    {
        $response = null;
        try {
            if (!empty($column) && !empty($value)) {
                $response = $this->student_fee_repository->findOneBy([$column => $value]);
            } else {
                if (!empty($FeeID)) {
                    $response = $this->student_fee_repository->find($FeeID);
                }
            }
        } catch (\Exception $exception) {
            $response = $exception->getMessage();
        }

        return $response;
    }

    ## delete student fee
    public function deleteStudentFee(int $param, $markAsDelete = true)
    {
        $response = [];
        $studentFee = $this->getFeeFromStudentFeeTable($param);
        if ($studentFee instanceof StudentFee) {
            ## delete student fee.
            try {
                if ($markAsDelete) {
                    ## mark as delete
                    $studentFee->setIsDeleted(true);
                    $this->entity_manager->persist($studentFee);
                    $this->entity_manager->flush();
                    $response = 'OK';
                } else {


                    $attachedClassFeeTemplate = count($studentFee->getClassFeeTemplates()->toArray());
                    $attachedClassFee = count($studentFee->getClassFees()->toArray());
                    $attachedClassFeeScheduler = count($studentFee->getClassFeeSchedules()->toArray());

                    /** @var ClassFeeTemplate */
                    if ($attachedClassFeeTemplate > 0) {
                        $response = $this->default_function->push_error(
                            $response,
                            'The '.$studentFee->getName(
                            ).' is attached with '.$attachedClassFeeTemplate.' Class template'
                        );
                    }
                    /** @var ClassFee */
                    if ($attachedClassFee > 0) {
                        $response = $this->default_function->push_error(
                            $response,
                            'The '.$studentFee->getName().' is attached with '.$attachedClassFee.' Class Fee'
                        );
                    }
                    /** @var ClassFeeSchedule */
                    if ($attachedClassFeeScheduler > 0) {
                        $response = $this->default_function->push_error(
                            $response,
                            'The '.$studentFee->getName(
                            ).' is attached with '.$attachedClassFeeScheduler.' Class Fee Scheduler'
                        );
                    }

                    if (empty($response)) {
                        ## delete from database
                        $this->entity_manager->remove($studentFee);
                        $this->entity_manager->flush();
                        $response = 'OK';
                    } else {
                        $response = $this->default_function->push_error(
                            $response,
                            'Please delete the attached fee before deleting this one.'
                        );
                    }

                }
            } catch (\Exception $exception) {
                $response = $exception->getMessage();
            }

        } else {
            $response = $studentFee;
        }

        return $response;
    }

    /**
     * Class Fee Template
     *
     * */
    ## get fee from the student fee table
    public function getFeeFromClassFeeTemplateTable(int $FeeID = null, string $key = null, $value = null)
    {
        $response = null;
        try {
            if (!empty($FeeID)) {
                $response = $this->class_fee_template_repository->find($FeeID);
            } else {
                if (!empty($key) && !empty($value)) {
                    $response = $this->class_fee_template_repository->findBy([$key => $value]);
                }
            }

        } catch (\Exception $exception) {
            $response = $exception->getMessage();
        }

        return $response;
    }

    ## add fee in the class_fee_template table.
    public function addFeeInClassFeeTemplateTable($feeList, ClassTemplate $class_template)
    {

        ## if fee list is empty
        if (empty($feeList)) {
            return 'Invalid input';
        }

        ## if fee list is not an array then make it array
        if (!is_array($feeList)) {
            $feeList = explode(' ', $feeList);
        }

        ## duplicate fee check
        $feeList = $this->class_fee_template_repository->bulkDuplicationCheck($feeList, $class_template->getId());

        ## save data in the table.
        foreach ($feeList as $key => $value) {
            try {
                $StudentFee = $this->getFeeFromStudentFeeTable($value);
                if ($StudentFee instanceof StudentFee) {
                    $ClassFeeTemplate = new ClassFeeTemplate();
                    $ClassFeeTemplate->setAmount($StudentFee->getAmount());
                    $ClassFeeTemplate->setFee($StudentFee);
                    $ClassFeeTemplate->setClassTemplate($class_template);

                    $this->entity_manager->persist($ClassFeeTemplate);
                    $this->entity_manager->flush();
                }
            } catch (\Exception $exception) {
                $this->errorArray = $this->default_function->push_error($this->errorArray, $exception->getMessage());
            }
        }

        ## sent back error, if we have it.
        if (empty($this->errorArray)) {
            return $this->errorArray;
        }

    }

    ## edit fee in the class_fee_template table.
    public function editFeeInClassFeeTemplateTable(ParameterBag $bag)
    {
        ## find the class fee template

        $ClassFeeTemplate = $this->getFeeFromClassFeeTemplateTable($bag->get('___rand__f'));
        if (!$ClassFeeTemplate instanceof ClassFeeTemplate) {
            return $ClassFeeTemplate;
        }

        try {
            $ClassFeeTemplate->setAmount($bag->get('amount'));
            $this->entity_manager->persist($ClassFeeTemplate);
            $this->entity_manager->flush();
        } catch (\Exception $exception) {
            $this->errorArray = $this->default_function->push_error($this->errorArray, $exception->getMessage());
        }

        if (empty($this->errorArray)) {
            return 'OK';
        } else {
            return $this->errorArray;
        }

    }

    ## delete ClassFeeTemplate fee
    public function deleteClassFeeTemplate(int $param, $markAsDeleted = true)
    {
        $response = [];
        $classFeeTemplate = $this->getFeeFromClassFeeTemplateTable($param);
        if ($classFeeTemplate instanceof ClassFeeTemplate) {

            try {

                if ($markAsDeleted) {
                    ## mark as delete
                    $classFeeTemplate->setIsDeleted(true);
                    $this->entity_manager->persist($classFeeTemplate);
                    $this->entity_manager->flush();
                    $response = 'OK';
                } else {
                    ## remove from database
                    $classFee = count($classFeeTemplate->getClassFees()->toArray());
                    if ($classFee > 0) {
                        $response = $this->default_function->push_error(
                            $response,
                            'This fee has attached with '.$classFee.' class of session'
                        );
                        $response = $this->default_function->push_error(
                            $response,
                            'Please delete class before delete in the session before deleting this one'
                        );

                    }
                    if (empty($response)) {
                        $this->entity_manager->remove($classFeeTemplate);
                        $this->entity_manager->flush();
                        $response = 'OK';

                    }
                }

            } catch (\Exception $exception) {
                $response = $exception->getMessage();
            }
        } else {
            $response = $classFeeTemplate;
        }

        return $response;
    }


    ## bulk delete class fee template
    ## single classTemplate has attached to multiple ClassfeeTemplate so, if we gone delete the classTemplate we have to delete all ClassFeeTemplate related to this one.
    ## so basically this method is used to delete the classFeeTemplate by reference of ClassTemplate
    public function deleteClassFeeTemplateByClassTemplateRef(ClassTemplate $classTemplate, $markAsDelete = true)
    {
        $classTemplateArray = $classTemplate->getClassFeeTemplates()->toArray();
        $response = [];
        /** @var ClassFeeTemplate $value */
        foreach ($classTemplateArray as $key => $value) {
            $response = $this->default_function->push_error(
                $response,
                $this->deleteClassFeeTemplate($value->getId(), $markAsDelete)
            );
        }

        return $this->default_function->purify_success_error_response_in_bulk_operation($response);
    }

    /**
     *
     * ===== Class Fee
     * @param int|null $FeeID
     * @param string|null $key
     * @param string|null $value
     * @return ClassFee|ClassFee[]|string|null
     */

    ## get fee from the student fee table
    public function getFeeFromClassFeeTable(int $FeeID = null, string $key = null, string $value = null)
    {
        try {
            if (!empty($FeeID)) {
                $response = $this->class_fee_repository->find($FeeID);
            } elseif (!empty($key) && !empty($value)) {
                $response = $this->class_fee_repository->findBy([$key => $value], ['is_deleted' => 'ASC']);
            }
        } catch (\Exception $exception) {
            $response = $exception->getMessage();
        }

        return $response;
    }

    ## save fee in the class fee table.
    public function addFeeInClassFeeTable(array $feeList, Classes $classes, $ClassEdit = false)
    {
        ## if fee list is empty
        if (empty($feeList)) {
            return 'Invalid input';
        }

        ## if fee list is not an array then make it array
        if (!is_array($feeList)) {
            $feeList = explode(' ', $feeList);
        }

        ## save data in the table.
        foreach ($feeList as $key => $value) {
            try {

                ## Get Class Fee
                $ClassFee = $this->entity_manager->getRepository('App:ClassFee')->findBy(
                    [
                        'fee' => $value,
                        'class' => $classes->getId(),
                        'is_deleted' => 0,
                    ]
                );

                if ($ClassFee instanceof ClassFee || (is_array($ClassFee) && !empty($ClassFee))) {
                    ## if Class fee already found
                    continue;
                } else {
                    if ($ClassEdit) {
                        ## edit a Class
                        $ClassFeeTemplate = $this->entity_manager->getRepository('App:ClassFeeTemplate')->findBy(
                            [
                                'fee' => $value,
                                'class_template' => $classes->getClassTemplate()->getId(),
                                'is_deleted' => 0,
                            ]
                        );
                    } else {
                        ## assigning Class to Session
                        $ClassFeeTemplate = $this->entity_manager->getRepository('App:ClassFeeTemplate')->find($value);
                    }
                    if ($ClassEdit) {
                        ## Class Fee Template Found,, now add Class Student Fee as Class Fee
                        $StudentFee = $this->getFeeFromStudentFeeTable($value);
                        if (!$ClassFeeTemplate instanceof ClassFeeTemplate) {
                            ## If Class Fee Template is not found - then save it into database
                            $ClassFeeTemplate = new ClassFeeTemplate();
                            $ClassFeeTemplate->setFee($StudentFee);
                            $ClassFeeTemplate->setClassTemplate($classes->getClassTemplate());
                            $ClassFeeTemplate->setAmount($StudentFee->getAmount());
                            $ClassFeeTemplate->setIsDeleted($StudentFee->getIsDeleted());
                            $this->entity_manager->persist($ClassFeeTemplate);
                            $this->entity_manager->flush();
                        }
                        if ($StudentFee instanceof StudentFee) {
                            ## if Student fee found then save all things into ClassFee.
                            $ClassFee = new ClassFee();
                            $ClassFee->setAmount($ClassFeeTemplate->getAmount());
                            $ClassFee->setFee($StudentFee);
                            $ClassFee->setClass($classes);
                            $ClassFee->setClassFeeTemplate($ClassFeeTemplate);
                            $this->entity_manager->persist($ClassFee);
                            $this->entity_manager->flush();
                        }
                    } else {
                        if ($ClassFeeTemplate instanceof ClassFeeTemplate) {
                            ## if Student fee found then save all things into ClassFee.
                            $ClassFee = new ClassFee();
                            $ClassFee->setAmount($ClassFeeTemplate->getAmount());
                            $ClassFee->setFee($ClassFeeTemplate->getFee());
                            $ClassFee->setClass($classes);
                            $ClassFee->setClassFeeTemplate($ClassFeeTemplate);
                            $this->entity_manager->persist($ClassFee);
                            $this->entity_manager->flush();
                        }
                    }
                }
            } catch (\Exception $exception) {
                $this->errorArray = $this->default_function->push_error($this->errorArray, $exception->getMessage());
            }
        }

        ## sent back error, if we have it.
        if (empty($this->errorArray)) {
            return $this->errorArray;
        }
    }

    ## edit fee in the class_fee_template table.
    public function editFeeInClassFeeTable(ParameterBag $bag)
    {
        ## find the class fee
        $ClassFee = $this->getFeeFromClassFeeTable($bag->get('___rand__f'));
        if (!$ClassFee instanceof ClassFee) {
            return $ClassFee;
        }

        try {
            $ClassFee->setAmount($bag->get('amount'));
            $this->entity_manager->persist($ClassFee);
            $this->entity_manager->flush();
        } catch (\Exception $exception) {
            $this->errorArray = $this->default_function->push_error($this->errorArray, $exception->getMessage());
        }

        if (empty($this->errorArray)) {
            return 'OK';
        } else {
            return $this->errorArray;
        }

    }

    ## delete fee from ClassFee table
    public function deleteClassFee(int $classFeeID, $markAsDeleted = true)
    {
        $response = [];
        $classFee = $this->getFeeFromClassFeeTable($classFeeID);
        if ($classFee instanceof ClassFee) {

            try {

                if ($markAsDeleted) {
                    ## mark as delete
                    $classFee->setIsDeleted(true);
                    $this->entity_manager->persist($classFee);
                    $this->entity_manager->flush();
                    $response = 'OK';
                } else {
                    ## remove from database
                    $classFeeSchedule = count($classFee->getClassFeeSchedules()->toArray());
                    if ($classFeeSchedule > 0) {
                        $response = $this->default_function->push_error(
                            $response,
                            'This fee has used  '.$classFee.' times to make Fee Schedule'
                        );
                    }

                    $classSessionEnrolmentScheduler = count(
                        $classFee->getClassSessionEnrolmentFeeSchedules()->toArray()
                    );
                    if ($classSessionEnrolmentScheduler > 0) {
                        $response = $this->default_function->push_error(
                            $response,
                            'This fee has used  '.$classSessionEnrolmentScheduler.' times to make Special Fee Schedule'
                        );
                    }

                    if (!empty($response)) {
                        $response = $this->default_function->push_error(
                            $response,
                            'Please delete class before delete in the session before deleting this one'
                        );
                    } else {
                        $this->entity_manager->remove($classFee);
                        $this->entity_manager->flush();
                        $response = 'OK';
                    }
                }

            } catch (\Exception $exception) {
                $response = $exception->getMessage();
            }
        } else {
            $response = $classFee;
        }

        return $response;
    }

    ## bulk delete class fee
    ## single class has attached to multiple fees so, if we gone delete the class we have to delete all fees related to this one.
    ## so basically this method is used to delete the classFee by reference of Class
    public function bulkDeleteClassFeeByClassRef(Classes $classes, $markAsDelete = true)
    {
        $classFeeArray = $classes->getClassFees()->toArray();
        $response = [];
        /** @var ClassFee $value */
        foreach ($classFeeArray as $key => $value) {
            $response = $this->default_function->push_error(
                $response,
                $this->deleteClassFee($value->getId(), $markAsDelete)
            );
        }

        return $this->default_function->purify_success_error_response_in_bulk_operation($response);
    }

}
