<?php

namespace App\Controller;

use App\Entity\ClassTemplate;
use App\Entity\EmailTemplates;
use App\Entity\GradeCategories;
use App\Entity\PaymentMethods;
use App\Entity\ScheduleMessage;
use App\Entity\SMSGateways;
use App\Entity\StudentGrades;
use App\Entity\Zones;
use App\Repository\AccountHeadsRepository;
use App\Repository\AppSettingsRepository;
use App\Repository\AttendanceStatusRepository;
use App\Repository\ClassesRepository;
use App\Repository\ClassTemplateRepository;
use App\Repository\EmailTemplatesRepository;
use App\Repository\PaymentMethodsRepository;
use App\Repository\ScheduleMessageRepository;
use App\Repository\SessionRepository;
use App\Repository\SMSGatewaysRepository;
use App\Repository\StudentFeeRepository;
use App\Repository\SubjectsRepository;
use App\Repository\UserRepository;
use App\Repository\ZonesRepository;
use App\Service\AppSettings;
use App\Service\Attendance\Base;
use App\Service\Constants;
use App\Service\DefaultFunction;
use App\Service\ScheduleMessagesService\Files;
use App\Service\SMSGateways\OperationalionalGatewayClass;
use App\Service\StudentClasses;
use App\Service\Students\Grade\ClassGradeService;
use App\Service\Students\Grade\GradeBase;
use App\Service\Students\Grade\GradeCategoriesService;
use App\Service\StudentSession;
use App\Service\SubjectService;
use App\Service\UserService;
use Doctrine\ORM\Query\QueryException;
use Knp\Snappy\Pdf;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Nzo\UrlEncryptorBundle\Annotations\ParamDecryptor;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\ParameterBag;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Intl\Locale;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Security\Core\Encoder\UserPasswordEncoderInterface;
use Symfony\Component\Translation\Loader\ArrayLoader;
use Symfony\Component\Translation\Loader\YamlFileLoader;
use Symfony\Component\Translation\Translator;
use Symfony\Contracts\Translation\TranslatorInterface;

/**
 * @IsGranted("ROLE_USER")
 * */
class AppSettingsController extends AbstractController
{

    /**
     * @var Request
     */
    private $request;
    /**
     * @var AppSettingsRepository
     */
    private $app_settings_repository;
    /**
     * @var AppSettings
     */
    private $app_settings_service;

    public function __construct(AppSettings $app_settings_service, AppSettingsRepository $app_settings_repository)
    {
        $this->app_settings_repository = $app_settings_repository;
        $this->app_settings_service = $app_settings_service;
    }

    /**
     *
     * @Route("/{_locale}/app/settings", name="app_settings", defaults={"_locale"="en"} )
     * @param Request $request
     * @param TranslatorInterface $translator
     * @param Constants $constants
     * @param AppSettings $app_settings
     *
     * @param ZonesRepository $zones_repository
     * @param AttendanceStatusRepository $attendance_status_repository
     *
     * @return Response
     * @throws QueryException
     */
    public function index(
        Request $request,
        TranslatorInterface $translator,
        Constants $constants,
        AppSettings $app_settings,
        ZonesRepository $zones_repository,
        AttendanceStatusRepository $attendance_status_repository
    ) {
        return $this->render(
            'app_settings/index.html.twig',
            [
                'selectedLocale' => $app_settings->getUserPreferLocal(),
                'zones' => $zones_repository->findAll(),
                'attendance_status' => $attendance_status_repository->findAll(),
                'app_settings' => $this->app_settings_repository->InArray(),
            ]
        );
    }


    /**
     *
     * @Route("/{_locale}/app/report/settings", name="app_reporting_settings", defaults={"_locale"="en"} )
     * PURPOSE: Using to view the pdf settings
     *
     * @param EmailTemplatesRepository $emailTemplatesRepository
     *
     * @return Response
     */
    public function pdfReportingSettings(EmailTemplatesRepository $emailTemplatesRepository)
    {
        return $this->render(
            'app_settings/pdfSettings.html.twig',
            [
                'emailTemplates' => $emailTemplatesRepository->findAll(),
            ]
        );
    }

    /**
     * @Route("{_locale}/app/settings/updateLocale/{query}", name="request_to_change_settings")
     * @param Request $request
     * @param AppSettings $app_settings
     * @param UserPasswordEncoderInterface $userPasswordEncoder
     * @param SubjectService $subjectService
     * @param StudentClasses $studentClasses
     * @param StudentSession $studentSession
     * @param ClassTemplateRepository $classTemplateRepository
     * @param SubjectsRepository $subjectsRepository
     * @param DefaultFunction $defaultFunction
     * @param null $query
     *
     * @return JsonResponse
     */
    public function ajaxHandler(
        Request $request,
        AppSettings $app_settings,
        UserPasswordEncoderInterface $userPasswordEncoder,
        SubjectService $subjectService,
        StudentClasses $studentClasses,
        StudentSession $studentSession,
        ClassTemplateRepository $classTemplateRepository,
        SubjectsRepository $subjectsRepository,
        DefaultFunction $defaultFunction,
        $query = null
    ) {
        $response = 'OK';
        if (empty($query)) {
            $query = $request->query->get('query');
        }

        if ($query == 'saveLocales') {
            ## save locale.
            $response = $app_settings->saveUserPreferLocal($request->request);
        } else {
            if ($query == 'saveSettings') {
                ## save general settings.
                $response = $app_settings->save_settings($request->request, $request->files);
            } else {
                if ($query == 'importData') {
                    $response = $this->app_settings_service->importDataFromOldFekara(
                        $request->request,
                        $userPasswordEncoder,
                        $defaultFunction,
                        $subjectService,
                        $studentClasses,
                        $studentSession,
                        $classTemplateRepository,
                        $subjectsRepository
                    );
                }
            }
        }

        return new JsonResponse($response, 200);
    }


    /**
     * =================== Check List
     * */

    /**
     * @Route("/{_locale}/app/settings/checklist/", name="app_settings_checklist", defaults={"_locale"="en"} )
     * @param Request $request
     *
     * @return Response
     */
    public function checkList(Request $request)
    {

        ## render html
        $response = $this->render('app_settings/checkList.html.twig', $this->app_settings_service->fekaraCheckList());
        if ($request->query->get('query') == 'checkListPoint') {
            ## insert basic data into database.
            $response = $this->app_settings_repository->insertBasicData();
            $response = new JsonResponse($response, 200);
        }

        return $response;
    }

    /**
     * =================== SMS GateWay Settings
     * */

    /**
     * @Route("/{_locale}/app/settings/sms-gateway/{gtwRef}", name="app_settings_sms_gateways",
     *                                                        defaults={"gtwRef"="null"} )
     * @ParamDecryptor(params={"gtwRef"})
     * @param OperationalionalGatewayClass $operationalionalGatewayClass
     *
     * @param Request $request
     * @param                              $gtwRef
     *
     * @return Response
     */
    public function showSMSGateWayForm(
        OperationalionalGatewayClass $operationalionalGatewayClass,
        Request $request,
        $gtwRef
    ) {
        $gateWay = $operationalionalGatewayClass->getGtw($gtwRef);
        if (!$gateWay instanceof SMSGateways) {
            $gateWay = new SMSGateways();
        }

        // if this is an ajax request.
        if ($request->isXmlHttpRequest()) {
            $returnResponse = new JsonResponse($operationalionalGatewayClass->saveGtw($request->request), 200);
        } else {
            $returnResponse = $this->render(
                'app_settings/smsGateways/add.html.twig',
                [
                    'smsGtw' => $gateWay,
                ]
            );
        }

        return $returnResponse;
    }

    /**
     * @Route("/{_locale}/app/settings/sms-gateway-list", name="app_settings_sms_gateways_list" )
     * @param SMSGatewaysRepository $SMSGatewaysRepository
     * PURPOSE: List of all SMS Gateways
     *
     * @return Response
     */
    public function SMSGateWayList(SMSGatewaysRepository $SMSGatewaysRepository)
    {
        return $this->render(
            'app_settings/smsGateways/list.html.twig',
            [
                'smsGtw' => $SMSGatewaysRepository->findAll(),
            ]
        );
    }


    /**
     * ================== Schedule Messages
     * */

    /**
     * @Route("/{_locale}/app/settings/scheduleMessage", name="app_settings_message_schedule")
     * @param UserService $userService
     *
     * @param Request $request
     * @param Files $files
     *
     * @param DefaultFunction $defaultFunction
     * @param \App\Service\ScheduleMessagesService\Base $base
     * @param ClassesRepository $classesRepository
     * @return Response
     */
    public function serverScheduleMessageForm(
        UserService $userService,
        Request $request,
        Files $files,
        DefaultFunction $defaultFunction,
        \App\Service\ScheduleMessagesService\Base $base,
        ClassesRepository $classesRepository
    ) {
        $targetedUrl = $request->query->get('targetedURL');
        $returnResponse = null;

        if (empty($targetedUrl)) {
            ## if the url where we have to download the pdf file, if empty then redirect to the homePage
//            $returnResponse = $this->redirect($defaultFunction->redirectToHomePage());
        } elseif ($request->isXmlHttpRequest() && count($request->request)) {
            ## submit form
            $returnResponse = $base->saveScheduleMessage($request->request);
            $returnResponse = new JsonResponse($returnResponse, 200);
        } else {

//            $files->downloadPdfFile($targetedUrl);
            $bag = new ParameterBag();
            $bag->set('lastOne', true);
            $bag->set('needURL', true);
            $bag->set('needPath', true);
//            $lastDownloadedFile = $files->getDownloadedFiles($bag);
            $lastDownloadedFile = null;
            $lastDownloadedFile_URL = empty($lastDownloadedFile['url']) ? '' : $lastDownloadedFile['url'];
            $lastDownloadedFile_Path = empty($lastDownloadedFile['path']) ? '' : $lastDownloadedFile['path'];

            ## if the url exits or available in the get variable then show the
            $returnResponse = $this->render(
                'schedulerMessages/index.html.twig',
                [
                    'students' => $userService->getStudents(),
                    'admins' => $userService->getAdmins(),
                    'class__list' => $classesRepository->getActiveClassesOfActiveSession(),
                    'lastDownloadedFile' => $lastDownloadedFile_URL,
                    'filePath' => $lastDownloadedFile_Path,
                ]
            );
        }

        return $returnResponse;
    }


    /**
     * @Route("/{_locale}/app/settings/schedule-messages-list/", name="app_settings_message_schedule_list")
     * @param ScheduleMessageRepository $scheduleMessageRepository
     * @return Response
     */
    public function serverScheduleMessages(ScheduleMessageRepository $scheduleMessageRepository)
    {
        return $this->render(
            'schedulerMessages/schedulerMessagesList.html.twig',
            [
                'scheduleMessages' => $scheduleMessageRepository->findAll(),
            ]
        );
    }

    /**
     * ================== Grades Management
     * */

    /**
     * @ParamDecryptor(params={"categoryID"})
     * @Route("/{_locale}/app/settings/GradeManagement/Category/{categoryID}", name="app_settings_grade_category",
     *                                                                         defaults={"categoryID":"null"})
     * @param GradeCategoriesService $gradeCategoriesService
     *
     * @param Request $request
     * @param                        $categoryID
     *
     * @return Response
     */
    public function showGradeCategoryForm(GradeCategoriesService $gradeCategoriesService, Request $request, $categoryID)
    {
        $GradeCategory = $gradeCategoriesService->getGradeCategory($categoryID);
        if (!$GradeCategory instanceof GradeCategories) {
            $GradeCategory = new GradeCategories();
        }
        // if this is an ajax request.
        if ($request->isXmlHttpRequest() && $request->isMethod('POST')) {
            try {
                $returnResponse = $gradeCategoriesService->saveGradeCategory($request->request);
            } catch (\Exception $exception) {
                $returnResponse = $exception->getMessage();
            }

            $returnResponse = new JsonResponse($returnResponse, 200);
        } else {
            $returnResponse = $this->render(
                'app_settings/grades/addEditGradeCategory.html.twig',
                [
                    'gradeCategory' => $GradeCategory,
                ]
            );
        }

        return $returnResponse;
    }

    /**
     * @Route("/{_locale}/app/settings/GradeManagement/Category-list", name="app_settings_grade_category_list")
     * @param GradeCategoriesService $gradeCategoriesService
     *
     * @return Response
     */
    public function showGradeCategoryList(GradeCategoriesService $gradeCategoriesService)
    {
        return $this->render(
            'app_settings/grades/gradeCategoryList.html.twig',
            [
                'gradeCategories' => $gradeCategoriesService->getGradeCategories(),
            ]
        );
    }

    /**
     * @Route("/{_locale}/app/settings/GradeManagement/add-grade", name="app_settings_assign_grade")
     * @param GradeBase $gradeBase
     *
     * @param Request $request
     *
     * @return string|JsonResponse|Response
     */
    public function addGradesInGradeCategoryForm(GradeBase $gradeBase, Request $request)
    {
        if ($request->isXmlHttpRequest() && $request->isMethod('POST')) {
            try {
                $returnResponse = $gradeBase->saveBulkGrades($request->request);
            } catch (\Exception $exception) {
                $returnResponse = $exception->getMessage();
            }
            $returnResponse = new JsonResponse($returnResponse, 200);
        } else {
            $returnResponse = $this->render(
                'app_settings/grades/assignGradesIntoGradeCategory.html.twig',
                [
                    'gradeCategories' => $gradeBase->getGradeCategories(),
                    'definedGrades' => $gradeBase::DefinedGrade,
                    'studentGrades' => [new StudentGrades()],
                ]
            );
        }

        return $returnResponse;
    }

    /**
     * @Route("/{_locale}/app/settings/GradeManagement/singleGradeItem", name="app_settings_singleGradeHtml")
     * @param GradeBase $gradeBase
     *
     * @param Request $request
     *
     * @return string|JsonResponse|Response
     */
    public function getSingleGradeItemHtml(GradeBase $gradeBase, Request $request)
    {

        $returnResponse = null;
        if ($request->isXmlHttpRequest() && $request->isMethod('POST')) {
            try {
                $returnResponse = 'escape__fromErro_'.$this->renderView(
                        'app_settings/grades/gradeSingleItem.html.twig',
                        [
                            'definedGrades' => $gradeBase::DefinedGrade,
                            'studentGrades' => [new StudentGrades()],
                        ]
                    );
            } catch (\Exception $exception) {
                $returnResponse = $exception->getMessage();
            }
        }
        $returnResponse = new JsonResponse($returnResponse, 200);

        return $returnResponse;
    }

    /**
     * @Route("/{_locale}/app/settings/GradeManagement/getGradeDefiniationsOnBaseOfGradeCategory", name="app_settings_getGradeDefinitionOnBaseOfGradeCategory")
     * @param ClassGradeService $classGradeService
     * @param Request $request
     * @return string|JsonResponse|Response
     */
    public function getGradeDefinitionOnDifferentBase(ClassGradeService $classGradeService, Request $request)
    {

        ## get grades list on base of grade Category
        if (!empty($request->request->get('gradeCat'))) {
            $gradesList = $classGradeService->getGrade(
                null,
                'student_grade_category',
                $request->request->get('gradeCat')
            );
            $filename = 'gradeSingleItem.html.twig';
        }

        ## get grade list on base of class Template
        if (!empty($request->request->get('classTemplateRef'))) {
            $gradesList = $classGradeService->getClassGradeTemplate(
                null,
                'class_template',
                $request->request->get('classTemplateRef')
            );
            $filename = 'classTemplate/editGrade.html.twig';
        }

        ## get grade list on base of Class.
        if (!empty($request->request->get('classRef'))) {
            $gradesList = $classGradeService->getClassGrade(
                null,
                'class',
                $request->request->get('classRef')
            );
            $filename = 'class/editGrade.html.twig';
        }

        ## if grade list empty then throw at lest one.
        if (empty($gradesList)) {
            $gradesList = [new StudentGrades()];
        }

        $returnResponse = null;
        // student_grade_category
        if ($request->isXmlHttpRequest() && $request->isMethod('POST')) {
            try {

                $returnResponse = 'escape__fromErro_'.$this->renderView(
                        'app_settings/grades/'.$filename,
                        [
                            'definedGrades' => $classGradeService::DefinedGrade,
                            'studentGrades' => $gradesList,
                            'postData' => $request->request,
                        ]
                    );
            } catch (\Exception $exception) {
                $returnResponse = $exception->getMessage();
            }
        }
        $returnResponse = new JsonResponse($returnResponse, 200);

        return $returnResponse;
    }


    /**
     * @Route("/{_locale}/app/settings/GradeManagement/classTemplate/grade-edit-form", name="app_settings_class_template_gradeEdit_form")
     * @param ClassGradeService $classGradeService
     * @param Request $request
     * @return string|JsonResponse|Response
     */
    public function editClassGradeTemplateForm(ClassGradeService $classGradeService, Request $request)
    {
        $returnResponse = null;
        // student_grade_category
        if ($request->isXmlHttpRequest() && $request->isMethod('POST')) {
            try {
                ## perform edition
                $returnResponse = $classGradeService->saveBulkGradeIntoClassTemplate($request->request);
            } catch (\Exception $exception) {
                $returnResponse = $exception->getMessage();
            }
        }
        $returnResponse = new JsonResponse($returnResponse, 200);

        return $returnResponse;
    }

    /**
     * @Route("/{_locale}/app/settings/GradeManagement/class/grade-edit-form", name="app_settings_class_gradeEdit_form")
     * @param ClassGradeService $classGradeService
     * @param Request $request
     * @return string|JsonResponse|Response
     */
    public function editClassGradeForm(ClassGradeService $classGradeService, Request $request)
    {
        $returnResponse = null;
        // student_grade_category
        if ($request->isXmlHttpRequest() && $request->isMethod('POST')) {
            try {
                ## perform edition
                $returnResponse = $classGradeService->saveBulkGradeIntoClass($request->request);
            } catch (\Exception $exception) {
                $returnResponse = $exception->getMessage();
            }
        }
        $returnResponse = new JsonResponse($returnResponse, 200);

        return $returnResponse;
    }

}

