<?php

namespace App\Controller;

use App\Entity\Accounts;
use App\Entity\Guardian;
use App\Entity\Invoices;
use App\Entity\Transactions;
use App\Repository\AccountHeadsRepository;
use App\Repository\AccountsRepository;
use App\Repository\ClassesRepository;
use App\Repository\InvoicesRepository;
use App\Repository\PaymentMethodsRepository;
use App\Repository\TransactionsRepository;
use App\Repository\UserRepository;
use App\Service\accounts\AccountsService;
use App\Service\accounts\InvoiceService;
use App\Service\accounts\PaymentsService;
use App\Service\accounts\TransactionService;
use App\Service\ApiHelpers;
use App\Service\AppSettings;
use App\Service\DefaultFunction;
use App\Service\Guardian\GuardianService;
use App\Service\SMSGateways\OperationalionalGatewayClass;
use App\Service\StudentClasses;
use App\Service\TriggerNotifications;
use App\Service\UserService;
use Doctrine\ORM\EntityManagerInterface;
use Knp\Bundle\SnappyBundle\KnpSnappyBundle;
use Knp\Bundle\SnappyBundle\Snappy\Response\PdfResponse;
use Knp\Snappy\Pdf;
use Nzo\UrlEncryptorBundle\Annotations\ParamDecryptor;
use Nzo\UrlEncryptorBundle\UrlEncryptor\UrlEncryptor;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\ParameterBag;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;

/**
 * @IsGranted("ROLE_USER")
 * */
class AccountsController extends AbstractController
{


    /**
     * @var AccountHeadsRepository
     */
    private $account_heads_repository;
    /**
     * @var AccountsRepository
     */
    private $accounts_repository;
    /**
     * @var PaymentMethodsRepository
     */
    private $payment_methods_repository;
    /**
     * @var AccountsService
     */
    private $accounts_service;
    /**
     * @var UserRepository
     */
    private $user_repository;
    /**
     * @var TransactionService
     */
    private $transaction_service;
    /**
     * @var InvoiceService
     */
    private $invoice_service;
    /**
     * @var PaymentsService
     */
    private $payments_service;
    /**
     * @var InvoicesRepository
     */
    private $invoices_repository;
    /**
     * @var ApiHelpers
     */
    private $api_helpers;
    /**
     * @var TransactionsRepository
     */
    private $transactions_repository;
    /**
     * @var UrlEncryptor
     */
    private $url_encryptor;
    /**
     * @var Pdf
     */
    private $pdf;
    /**
     * @var AppSettings
     */
    private $appSettings;

    public function __construct(
        UrlEncryptor $url_encryptor,
        AppSettings $appSettings,
        TransactionsRepository $transactions_repository,
        ApiHelpers $api_helpers,
        PaymentsService $payments_service,
        InvoicesRepository $invoices_repository,
        AccountHeadsRepository $account_heads_repository,
        InvoiceService $invoice_service,
        TransactionService $transaction_service,
        AccountsRepository $accounts_repository,
        PaymentMethodsRepository $payment_methods_repository,
        AccountsService $accounts_service,
        UserRepository $user_repository,
        Pdf $pdf
    ) {
        $this->account_heads_repository = $account_heads_repository;
        $this->accounts_repository = $accounts_repository;
        $this->payment_methods_repository = $payment_methods_repository;
        $this->accounts_service = $accounts_service;
        $this->user_repository = $user_repository;
        $this->transaction_service = $transaction_service;
        $this->invoice_service = $invoice_service;
        $this->payments_service = $payments_service;
        $this->invoices_repository = $invoices_repository;
        $this->api_helpers = $api_helpers;
        $this->transactions_repository = $transactions_repository;
        $this->url_encryptor = $url_encryptor;
        $this->pdf = $pdf;
        $this->appSettings = $appSettings;
    }


    /**
     * @Route("/{_locale}/accounts/list", name="accounts_list")
     * @IsGranted("ROLE_ACCOUNTANT")
     */
    public function account__list()
    {
        return $this->render(
            'accounts/accounts.html.twig',
            [
                'account__list' => $this->accounts_repository->findAll(),
            ]
        );
    }

    /**
     * @Route("/{_locale}/accounts/payment-methods", name="payment_methods")
     * @IsGranted("ROLE_ACCOUNTANT")
     */
    public function payment__methods()
    {
        return $this->render(
            'accounts/payment_methods.html.twig',
            [
                'payment_method_list' => $this->payment_methods_repository->findAll(),
            ]
        );
    }

    /**
     * @Route("/{_locale}/accounts/heads", name="account_heads")
     * @IsGranted("ROLE_ACCOUNTANT")
     */
    public function account_heads()
    {
        return $this->render(
            'accounts/account_heads.html.twig',
            [
                'account_heads_list' => $this->account_heads_repository->findAll(),
            ]
        );
    }


    /**
     * ======= Transactions
     *
     * */

    /**
     * @Route("/{_locale}/accounts/transactions/expense/{id}", name="transactions_expense", defaults={"id"=null},
     *                                                         requirements={"id"="\d+"})
     * @param null $id
     * @IsGranted("ROLE_ACCOUNTANT")
     * @return Response
     */
    public function transactions($id = null)
    {

        $transEntity = null;

        if (!empty($id)) {
            $transEntity = $this->transaction_service->get_the__transaction($id);
        }

        if (!$transEntity instanceof Transactions) {
            $transEntity = new Transactions();
        }

        return $this->render(
            'accounts/transaction_as_expense.html.twig',
            [
                ## list the of the accounts
                'account_list' => $this->accounts_repository->findAll(),
                ## list of the expense
                'expense_head' => $this->account_heads_repository->findBy(['ah_type' => false]),
                ## users list.
                'user_list' => $this->user_repository->findAll(),
                ## payment methods.
                'payment_method' => $this->payment_methods_repository->findAll(),
                ## transaction object
                'trans_ob' => $transEntity,
            ]
        );
    }

    /**
     * @Route("/{_locale}/accounts/transactions/list", name="transaction_list")
     *
     * @param Request $request
     * @IsGranted("ROLE_ACCOUNTANT")
     * @return Response
     */
    public function transaction_list(Request $request, DefaultFunction $defaultFunction)
    {
        $Bag = new ParameterBag();
        $Bag->set('is_expense_or_income', $request->get('amountInOut'));
        $Bag->set('trans_description', $request->get('transDesc'));
        $Bag->set('trans_id', $request->get('transID'));
        $Bag->set('transactionDateRange', $request->get('dateRange'));
        $Bag->set('trans_amount_id', $request->get('transAmount'));
        $Bag->set('payment_method', $request->get('pamentMethod'));
        $Bag->set('registerUser', $request->get('SystemUser'));
        $Bag->set('unRegisterUser', $request->get('in__v_UnRegUser'));

        $sendingData = [
            'trans__ls' => $this->transactions_repository->advanceSearch($Bag),
        ];

        ## print pdf file
        if ($request->query->get('printResult') == 1) {

            $filename = 'Transactions Report';
            $reportFileName = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('Transactions')
            );
            ## if report name is missing.
            if (empty($reportFileName)) {
                $reportFileName = 'pdfPrints/accounts.root/Transactions/default.html.twig';
            }

            ## render html file with data
            $html_fe = $this->renderView($reportFileName, $sendingData);

            return new Response(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );

        }

        $sendingData['trans__fullList'] = $this->transactions_repository->findAll();
        $sendingData['user_list'] = $this->user_repository->findAll();
        $sendingData['PaymentMethod'] = $this->payment_methods_repository->findAll();

        return $this->render('accounts/transaction__list.html.twig', $sendingData);
    }

    /**
     * @Route("/{_locale}/accounts/pay/invoice/{id}", name="pay__invoice",  defaults={"id"=null} )
     *
     * @param null $id
     * @ParamDecryptor(params={"id"})
     * @IsGranted("ROLE_ACCOUNTANT")
     * @return Response
     */
    public function pay___invoice($id = null)
    {

        $invoiceEntity = null;

        if (!empty($id)) {
            $invoiceEntity = $this->invoice_service->get__the__invoice($id);
        }

        if (!$invoiceEntity instanceof Invoices) {
            return $this->redirectToRoute('invoices_list');
        }


        return $this->render(
            'accounts/pay_invoice.html.twig',
            [
                'invoice' => $invoiceEntity,
                'payment_method' => $this->payment_methods_repository->findAll(),
                ## list of the incomes.
                'income_head' => $this->account_heads_repository->findBy(['ah_type' => true]),
                ## list the of the accounts
                'account_list' => $this->accounts_repository->findAll(),
            ]
        );
    }


    /**
     *  ====== Invoice
     * */


    /**
     * @Route("/{_locale}/accounts/add/invoice/{id}", name="invoices_", defaults={"id"=null})
     * @param null $id
     * @ParamDecryptor({"id"})
     * @IsGranted("ROLE_ACCOUNTANT")
     * @return Response
     */
    public function invoices__plus__invoice_item($id = null)
    {

        $InvEntity = null;

        if (!empty($id)) {
            $InvEntity = $this->invoice_service->get__the__invoice($id);
        }

        if (!$InvEntity instanceof Invoices) {
            $InvEntity = new Invoices();
        }

        return $this->render(
            'accounts/invoices.html.twig',
            [
                ## users list.
                'user_list' => $this->user_repository->findAll(),
                ## payment methods.
                'payment_method' => $this->payment_methods_repository->findAll(),
                ## invoice object
                'inv_ob' => $InvEntity,
            ]
        );
    }

    /**
     * @Route("/{_locale}/accounts/invoice/list/{sendMessage}", name="invoices_list")
     * @ParamDecryptor(params={"invoicePaidTo"})
     * @param Request $request
     * @param ClassesRepository $classesRepository
     *
     * @param null $sendMessage
     * @param OperationalionalGatewayClass $operationalionalGatewayClass
     * @return Response
     * @IsGranted("ROLE_ACCOUNTANT")
     */
    public function invoice__list(
        Request $request,
        ClassesRepository $classesRepository,
        $sendMessage = null,
        TriggerNotifications $triggerNotifications
    ) {

        $inv_ls = $this->invoices_repository->search__inv($request->query);

        if ($sendMessage == 1) {
            foreach ($inv_ls as $invoice) {
                /** @var Invoices $invoice */
                $invoice->sendSMSAboutInvoice($triggerNotifications);
            }
        }


        return $this->render(
            'accounts/invoices__list.html.twig',
            [
                'inv_ls' => $inv_ls,
                ## invoice list
                'invoice__list' => $this->invoices_repository->findAll(),
                ## list the of the accounts
                'account_list' => $this->accounts_repository->findAll(),
                ## list of the incomes.
                'income_head' => $this->account_heads_repository->findBy(['ah_type' => true]),
                ## users list.
                'user_list' => $this->user_repository->findAll(),
                ## payment methods.
                'payment_method' => $this->payment_methods_repository->findAll(),
                ## trans object
                'Classes' => $classesRepository->findAll(),
            ]
        );
    }

    /**
     * @Route("/{_locale}/accounts/guardian/invoice/list/", name="invoices_list_guardian")
     * @IsGranted("ROLE_GUARDIAN")
     * @param Request $request
     * @param ClassesRepository $classesRepository
     * @return Response
     */
    public function invoiceListForGuardian(Request $request, ClassesRepository $classesRepository)
    {
        $studentIds = GuardianService::getAttachedStudentsIdWithGuardian($this->getUser());
        $request->query->set('isGuardian', true);
        $request->query->set('guardianStudents', $studentIds);

        return $this->render(
            'accounts/Guardian/invoiceList.html.twig',
            [
                'inv_ls' => $this->invoices_repository->search__inv($request->query),
                ## trans object
                'Classes' => $classesRepository->findAll(),
            ]
        );
    }

    /**
     * @Route("/{_locale}/accounts/student/invoice/list/", name="invoices_list_student")
     * @IsGranted("ROLE_STUDENT")
     * @param Request $request
     * @param ClassesRepository $classesRepository
     * @return Response
     */
    public function invoiceListForStudent(Request $request)
    {
        $request->query->set('user', $this->getUser()->getId());

        return $this->render(
            'accounts/Student/invoiceList.html.twig',
            [
                'inv_ls' => $this->invoices_repository->search__inv($request->query),
            ]
        );
    }

    /**
     * @Route("/{_locale}/accounts/invoice/view/{invoice}", name="invoices_view", defaults={"invoice":null})
     * @ParamDecryptor(params={"invoice"})
     * @param Request $request
     * @param                 $invoice
     * @param DefaultFunction $defaultFunction
     * @IsGranted("ROLE_ACCOUNTANT")
     * @return PdfResponse|RedirectResponse|Response
     */
    public function viewInvoice(Request $request, $invoice, DefaultFunction $defaultFunction)
    {

        $templatePath = $defaultFunction->parseTemplatePathFromFullPath(
            $this->appSettings->getAppSettingsFromSessions('AdmissionInvoice')
        );

        if (empty($templatePath)) {
            ## default path
            $templatePath = 'pdfPrints/accounts.root/Admission.Invoice/default.html.twig';
        }

        $invoice = $this->invoice_service->get__the__invoice($invoice);
        if (!$invoice instanceof Invoices) {
            return $this->redirectToRoute('invoices_list');
        }

        ## data sent to the page
        $sendingData = [
            'invoice' => $invoice,
        ];

        ## print pdf file
        if ($request->query->get('printResult') == 1) {
            $html_fe = $this->renderView($templatePath, $sendingData);

            $filename = 'Invoice Report';

            return new Response(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );

        }

        return $this->render($templatePath, $sendingData);
    }

    /**
     * @Route("/{_locale}/accounts/payment/{id}", name="account_payment", defaults={"id"=null})
     * @param null $id
     * @IsGranted("ROLE_ACCOUNTANT")
     * @return Response
     */
    public function transactions__plus__invoice__plus__invoice_item($id = null)
    {

        $transEntity = null;

        if (!empty($id)) {
            $transEntity = $this->transaction_service->get_the__transaction($id);
        }

        if (!$transEntity instanceof Transactions) {
            ## create new payment
            $transEntity = new Transactions();
        } else {
            ## update an exiting payment if this transaction is instant payment
            if (!$transEntity->getIsInstantPayment()) {
                $transEntity = new Transactions();
            }
        }

        return $this->render(
            'accounts/instant_payment.html.twig',
            [
                ## list the of the accounts
                'account_list' => $this->accounts_repository->findAll(),
                ## list of the incomes
                'income_head' => $this->accounts_service->get_the_income_head(),
                ## users list.
                'user_list' => $this->user_repository->findAll(),
                ## payment methods.
                'payment_method' => $this->payment_methods_repository->findAll(),
                ## transaction object
                'trans_ob' => $transEntity,
            ]
        );
    }

    /**
     * @Route("{_locale}/accounts/requestHandler", name="requestHandler___accounts")
     * @param Request $request
     * @IsGranted("ROLE_ACCOUNTANT")
     * @return JsonResponse
     */
    public function ajaxHandler(Request $request)
    {
        $response = 'OK';

        ## get the query variable from the get request.
        $query = $request->query->get('query');

        switch ($query) {
            case 'add__edit__accounts':
                ## add, edit accounts.
                $response = $this->accounts_service->_add_update_accounts($request->request);
                break;
            case 'add__edit__account_head':
                ## add, edit accounts heads
                $response = $this->accounts_service->_add_update_accounts__head($request->request);
                break;
            case 'add__edit__payment_methods':
                ## add, edit payment methods.
                $response = $this->accounts_service->_add_update_payment_methods($request->request);
                break;
            case 'mani___invoice':
                ## add or update invoices.
                $response = $this->invoice_service->add_edit_invoice($request->request);
                break;
            case 's__tran__':
                ## add, update transaction
                $response = $this->transaction_service->_add_edit_transactions($request->request, $request->files);
                break;
            case 'a_pay__ru':
                ## add payment
                $response = $this->payments_service->add__update_payment($request->request, $request->files);
                break;
            case 's__inv':
                ## search invoice
                $response = $this->invoices_repository->search__inv($request->request);
                $response = 'escape__fromErro_'.$this->renderView(
                        'accounts/Invoices/invoice__listTable.html.twig',
                        [
                            'inv_ls' => $response,
                        ]
                    );
                break;
            case 'deleteInvoiceItem':
                ## delete invoice items
                $response = $this->invoice_service->deleteInvoiceItem(
                    $this->url_encryptor->decrypt($request->request->get('ref'))
                );
                break;
        }

        return new JsonResponse($response, 200);
    }

    /**
     * @Route("{_locale}/accouts/print-challan/{invoice}", name="print_invoice_and_challan_form",
     *                                                     defaults={"invoice":null})
     * @ParamDecryptor(params={"invoice"})
     * Print invoice challan form
     *
     * @param Request $request
     * @param                 $invoice
     * @param DefaultFunction $defaultFunction
     * @IsGranted("ROLE_ACCOUNTANT")
     * @return PdfResponse|RedirectResponse|Response
     */
    public function invoiceChallanForm(Request $request, $invoice, DefaultFunction $defaultFunction)
    {

        $invoice = $this->invoice_service->get__the__invoice($invoice);
        if (!$invoice instanceof Invoices) {
            return $this->redirectToRoute('invoices_list');
        }

        ## data sent to the page
        $sendingData = [
            'invoice' => $invoice,
        ];

        $templatePath = $defaultFunction->parseTemplatePathFromFullPath(
            $this->appSettings->getAppSettingsFromSessions('Challan')
        );

        if (empty($templatePath)) {
            ## default path
            $templatePath = 'pdfPrints/accounts.root/Challan/default.html.twig';
        }

        ## print pdf file
        if ($request->query->get('printResult') == 1) {
            $html_fe = $this->renderView($templatePath, $sendingData);
            $filename = 'Fee-Challan';

            return new Response(
                $this->pdf->getOutputFromHtml(
                    $html_fe,
                    [
                        'orientation' => 'landscape',
                    ]
                ), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );

        }

        return $this->render($templatePath, $sendingData);
    }

    /**
     * @Route("{_locale}/accouts/mark-invoice-cancel/{invoice}", name="mark_invoice_cancel")
     * @ParamDecryptor(params={"invoice"})
     */
    public function markInvoiceCancel($invoice, EntityManagerInterface $entityManager)
    {
        try {
            $invoice = $entityManager->getRepository(Invoices::class)->find($invoice);
            if ($invoice instanceof Invoices) {
                $invoice->setIsCancel(true);
                $entityManager->persist($invoice);
                $entityManager->flush();
            }
        } catch (\Exception $exception) {

        }

        return $this->redirectToRoute('invoices_list');
    }


    /**
     *  =================== Reporting ================
     * */

    /**
     * @Route("{_locale}/accounts/income/report", name="income_report")
     * Print the income report
     *
     * @param Request $request
     * @param DefaultFunction $defaultFunction
     * @IsGranted("ROLE_ACCOUNTANT")
     * @return Response
     */
    public function incomeReport(Request $request, DefaultFunction $defaultFunction)
    {
        $bag = new ParameterBag();
        $bag->set('payment_method', $this->url_encryptor->decrypt($request->query->get('paymentMethod')));
        $bag->set('account_head', $this->url_encryptor->decrypt($request->query->get('incomeHead')));
        $bag->set('transactionDateRange', $request->query->get('date'));
        $bag->set('is_expense_or_income', 2);

        $transactionsAsIncome = null;
        ## send no default by default.
//        if (count($request->query) > 0) {
        $transactionsAsIncome = $this->transactions_repository->advanceSearch($bag);
//        }
        ## if user search then go to the database to search data.
        $sendingData = [
            'transactionsAsIncome' => $transactionsAsIncome,
        ];

        ## print pdf report
        if ($request->query->get('printResult') == 1) {
            $filename = 'Income Report';
            $reportFileName = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('Income')
            );
            ## if report name is missing.
            if (empty($reportFileName)) {
                $reportFileName = 'pdfPrints/accounts.root/Income/default.html.twig';
            }

            ## sending details of filters in the report.
            $sendingData['paymentMethod'] = $this->payment_methods_repository->find($bag->get('payment_method'));
            $sendingData['accountHead'] = $this->account_heads_repository->find($bag->get('account_head'));

            ## render html file with data
            $html_fe = $this->renderView($reportFileName, $sendingData);

            return new Response(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );
        }

        $sendingData['incomeHead'] = $this->accounts_service->get_the_income_head();
        $sendingData['paymentMethod'] = $this->payment_methods_repository->findAll();

        return $this->render('accounts/reporting/income_report.html.twig', $sendingData);
    }

    /**
     * @Route("{_locale}/accouts/expense/report", name="expense_report")
     * Print invoice expense  report
     * @IsGranted("ROLE_ACCOUNTANT")
     * @param Request $request
     * @param DefaultFunction $defaultFunction
     *
     * @return Response
     */
    public function expenseReport(Request $request, DefaultFunction $defaultFunction)
    {
        $bag = new ParameterBag();
        $bag->set('payment_method', $this->url_encryptor->decrypt($request->query->get('paymentMethod')));
        $bag->set('account_head', $this->url_encryptor->decrypt($request->query->get('expenseHead')));
        $bag->set('transactionDateRange', $request->query->get('date'));
        $bag->set('is_expense_or_income', 1);
        $transactionsAsIncome = $this->transactions_repository->advanceSearch($bag);

//		$transactionsAsIncome = null;
//		if ( count( $request->query ) > 0 ) {
//            $transactionsAsIncome = $this->transactions_repository->advanceSearch( $bag );
//        }

        $sendingData = [
            'transactionsAsIncome' => $transactionsAsIncome,
        ];

        ## print pdf report
        if ($request->query->get('printResult') == 1) {
            $filename = 'Expense Report';
            $reportFileName = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('Expense')
            );
            ## if report name is missing.
            if (empty($reportFileName)) {
                $reportFileName = 'pdfPrints/accounts.root/Expense/default.html.twig';
            }


            ## sending details of filters in the report.
            $sendingData['paymentMethod'] = $this->payment_methods_repository->find($bag->get('payment_method'));
            $sendingData['expenseHead'] = $this->account_heads_repository->find($bag->get('account_head'));

            ## render html file with data
            $html_fe = $this->renderView($reportFileName, $sendingData);

            return new Response(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );
        }

        $sendingData['expenseHead'] = $this->accounts_service->get_the_expense_head();
        $sendingData['paymentMethod'] = $this->payment_methods_repository->findAll();

        return $this->render('accounts/reporting/expense_report.html.twig', $sendingData);
    }

    /**
     * @Route("{_locale}/accouts/defaulter/report", name="defaulter_report")
     * Print invoice expense  report
     * @IsGranted("ROLE_ACCOUNTANT")
     * @param Request $request
     * @param DefaultFunction $defaultFunction
     *
     * @return Response
     */
    public function DefaulterReport(Request $request, DefaultFunction $defaultFunction)
    {

        $bag = new ParameterBag();
        $bag->set('invStatus', 'defaulter');
        $bag->set('overDue', $request->query->get('dueDate'));

        $defaulterReport = null;
        if (count($request->query)) {
            $defaulterReport = $this->invoices_repository->search__inv($bag);
        }

        $sendingData = [
            'defaulterReport' => $defaulterReport,
        ];

        ## print pdf report
        if ($request->query->get('printResult') == 1) {
            $filename = 'Defaulter Report';
            $reportFileName = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('Defaulter')
            );
            ## if report name is missing.
            if (empty($reportFileName)) {
                $reportFileName = 'pdfPrints/accounts.root/Defaulter/default.html.twig';
            }
            ## render html file with data
            $html_fe = $this->renderView($reportFileName, $sendingData);

            return new Response(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );
        }

        return $this->render('accounts/reporting/defaulter_report.html.twig', $sendingData);
    }
}
