<?php

namespace App\EventSubscriber;

use App\Service\AppSettings;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Event\ControllerEvent;
use Symfony\Component\HttpKernel\Event\FilterControllerEvent;
use Symfony\Component\HttpKernel\Event\GetResponseEvent;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Security\Core\Security;

class LocaleSubscriber implements EventSubscriberInterface {

	private $defaultLocale;
	/**
	 * @var AppSettings
	 */
	private $app_settings;
	/**
	 * @var Security
	 */
	private $security;
	/**
	 * @var UrlGeneratorInterface
	 */
	private $url_generator;

	/**
	 * LocaleSubscriber constructor.
	 *
	 * @param string      $defaultLocale
	 * @param AppSettings $app_settings
	 */
	public function __construct( $defaultLocale = 'en', AppSettings $app_settings, Security $security, UrlGeneratorInterface $url_generator ) {
		$this->defaultLocale = $defaultLocale;
		$this->app_settings  = $app_settings;
		$this->security      = $security;
		$this->url_generator = $url_generator;
	}

	/**
	 * @var GetResponseEvent $event
	 * PURPOSE: On kernel request
	 * */
	public function onKernetRequest( $event ) {

		/**
		 * @var Request $request
		 * */
		$request = $event->getRequest();
		if ( ! $request->hasPreviousSession() ) {
			return;
		}

		## set timezone
		$this->setTimeZoneBasedOnSettings();

		## get locale.
		if ( $locale = $request->attributes->get( '_locale' ) ) {
			$request->getSession()->set( '_locale', $locale );
		} else {
			$request->setLocale( $request->getSession()->get( '_locale', $this->defaultLocale ) );
		}

	}

	## on each controller request
	public function onKernelController( ControllerEvent $controller_event ) {
		$controller = $controller_event->getController();

		// when a controller class defines multiple action methods, the controller
		// is returned as [$controllerInstance, 'methodName']
		if ( is_array( $controller ) ) {
			$controller = $controller[0];
		}
	}

	public static function getSubscribedEvents() {
		return [
			KernelEvents::REQUEST    => [
				[ 'onKernetRequest', 20 ],
			],
			KernelEvents::CONTROLLER => 'onKernelController',
		];
	}


	/*
	 * Setting timezone based on the application settings.
	 * */
	private function setTimeZoneBasedOnSettings() {
		## fetch timezone from the settings.
		$timezone = $this->app_settings->getAppSettingsFromSessions( 'time_zone' );
		## set the timezone.
		if ( ! empty( $timezone ) ) {
			date_default_timezone_set( $timezone );
		}
	}
}
