<?php

/*
 *  column -- attendance_type: 1 mean class has subject wise attendance and 0 mean class has class wise attendance
 * */

namespace App\Repository;

use App\Entity\Classes;
use App\Entity\ClassSubjectTemplate;
use App\Entity\User;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\ORM\NonUniqueResultException;
use PhpParser\Node\Param;
use Symfony\Bridge\Doctrine\RegistryInterface;
use Symfony\Component\HttpFoundation\ParameterBag;

/**
 * @method Classes|null find($id, $lockMode = null, $lockVersion = null)
 * @method Classes|null findOneBy(array $criteria, array $orderBy = null)
 * @method Classes[]    findAll()
 * @method Classes[]    findBy(array $criteria, array $orderBy = null, $limit = null, $offset = null)
 */
class ClassesRepository extends ServiceEntityRepository
{

    /**
     * @var RegistryInterface
     */
    private $registry;
    /**
     * @var AllocateTeacherToClassAndSubjectsRepository
     */
    private $allocateTeacherToClassAndSubjectsRepository;

    public function __construct(
        RegistryInterface $registry,
        AllocateTeacherToClassAndSubjectsRepository $allocateTeacherToClassAndSubjectsRepository
    ) {
        parent::__construct($registry, Classes::class);
        $this->registry = $registry;
        $this->allocateTeacherToClassAndSubjectsRepository = $allocateTeacherToClassAndSubjectsRepository;
    }

    ## delete all template which have this specific session id.
    public function deleteAllClassesAgaintSpecificSession($session_id)
    {
        $connection = $this->getEntityManager()->getConnection();
        $sql = 'DELETE FROM classes where attached_sessions_id = ?';
        $statement = $connection->prepare($sql);
        $statement->bindValue(1, $session_id);
        $statement->execute();
    }

    ## get classes of the active session from the classes table.

    /**
     * @param User| null $currentLoggedInUser
     * @return \Doctrine\ORM\QueryBuilder
     */
    public function getClassesOfActiveSessionQueryBuilder(User $currentLoggedInUser = null)
    {
        $queryBuilder = $this->createQueryBuilder('classes')
            ->leftJoin('classes.session', 'session')
            ->addSelect('session')
            ->where('session.active= :active__')
            ->setParameter('active__', 1)->andWhere('session.is_deleted = 0');

        ## server only those classes in which teacher is enrolled
        if ($currentLoggedInUser instanceof User && in_array('ROLE_TEACHER', $currentLoggedInUser->getRoles())) {
            $allocatedClassesToTeacher = $this->allocateTeacherToClassAndSubjectsRepository->getUniqueClassAllocationByTeacherID(
                $currentLoggedInUser->getId()
            );
            if (is_array($allocatedClassesToTeacher) && !empty($allocatedClassesToTeacher)) {
                $queryBuilder->andWhere($queryBuilder->expr()->in('classes.id', $allocatedClassesToTeacher));
            }
        }


        return $queryBuilder;

    }

    ## get classes of the active session from the classes table.

    /**
     * @param bool $inArray
     *
     * @param User|null $currentLoggedInUser
     * @return \Doctrine\ORM\Query | array
     */
    public function getClassesOfActiveSession(bool $inArray = false, User $currentLoggedInUser = null)
    {
        $queryBuilder = $this->getClassesOfActiveSessionQueryBuilder($currentLoggedInUser)->getQuery();
        if ($inArray) {
            ## get result in array form
            $queryBuilder = $queryBuilder->getArrayResult();
        } else {
            ## get result in object
            $queryBuilder = $queryBuilder->getResult();
        }


        return $queryBuilder;
    }

    /**
     * @param bool $inArray
     *
     * @return \Doctrine\ORM\Query | array
     */
    public function getActiveClassesOfActiveSession(bool $inArray = false)
    {
        $queryBuilder = $this->getClassesOfActiveSessionQueryBuilder();
        $queryBuilder = $queryBuilder->andWhere('classes.active = 1 and classes.is_deleted = 0')->getQuery();

        if ($inArray) {
            ## get result in array form
            $queryBuilder = $queryBuilder->getArrayResult();
        } else {
            ## get result in object
            $queryBuilder = $queryBuilder->getResult();
        }

        return $queryBuilder;
    }

    ## get all classes
    public function getAllClasses(User $currentLoggedInUser = null)
    {
        $queryBuilder = $this->createQueryBuilder('classes');

        ## server only those classes in which teacher is enrolled
        if ($currentLoggedInUser instanceof User && in_array('ROLE_TEACHER', $currentLoggedInUser->getRoles())) {
            $allocatedClassesToTeacher = $this->allocateTeacherToClassAndSubjectsRepository->getUniqueClassAllocationByTeacherID(
                $currentLoggedInUser->getId()
            );
            if (is_array($allocatedClassesToTeacher) && !empty($allocatedClassesToTeacher)) {
                $queryBuilder->andWhere($queryBuilder->expr()->in('classes.id', $allocatedClassesToTeacher));
            }
        }

        return $queryBuilder->getQuery()->getResult();

    }


    /*
     * ATTENDANCE - related.
     * */

    ## some classes offers subject wise attendance & some of them are offering class wise attendance.
    ## this method get class the list of those classes whose attendance is subject based
    public function getSubjectWiseAttendanceClassesList()
    {
        $queryBuilder = $this->getClassesOfActiveSessionQueryBuilder();
        ## 1 means subject wise attendance is true.
        $queryBuilder->andWhere('classes.attendance_type = :attendance_type')->setParameter('attendance_type', 1);

        return $queryBuilder->getQuery()->getResult();
    }


    ## some classes offers subject wise attendance & some of them are offering class wise attendance.
    ## this method get class the list of those classes whose attendance is class based.
    public function getClassWiseAttendanceClassesList()
    {
        $queryBuilder = $this->getClassesOfActiveSessionQueryBuilder();
        ## 0 mean not subject wise attendance.
        $queryBuilder->andWhere('classes.attendance_type = :attendance_type')
            ->orWhere('classes.attendance_type is null')
            ->setParameter('attendance_type', 0);

        return $queryBuilder->getQuery()->getResult();
    }


    ## get present and absent student no of student of active class with sessions.
    ## this method return an array of the class id as key and not of present user in the class as value.
    public function getNoOfPresentStudentInActiveClass($date = null, $classID = null)
    {
        ## return the query of active classes
        $queryBuilder = $this->getClassesOfActiveSessionQueryBuilder();

        ## class session enrolment.
        $queryBuilder = $queryBuilder->leftJoin('classes.classSessionEnrolments', 'class_session_enrolments')
            ->addSelect('class_session_enrolments')
            ->leftJoin('class_session_enrolments.attendances', 'attendances')
            ->addSelect('attendances');

        ## 1 for present and 0 for absent
        $queryBuilder->andwhere('attendances.attendance_status_type = 1');

        ## class attendance
        if (!empty($classID)) {
            $queryBuilder->andWhere('classes = :classRef')->setParameter('classRef', $classID);
        }

        ## duration filter
        if (!empty($date) && strpos($date, ' / ')) {
            // if date range is given
            $attendanceDate = explode(' / ', $date);
            $queryBuilder->andwhere('attendances.created BETWEEN :attendanceFrom and :attendanceTo');
            $queryBuilder->setParameter('attendanceFrom', date('Y-m-d', strtotime(trim($attendanceDate[0]))).'%');
            $queryBuilder->setParameter('attendanceTo', date('Y-m-d', strtotime(trim($attendanceDate[1]))).'%');
        } else {
            if (!empty($date)) {
                // if single date is given
                $queryBuilder->andwhere('attendances.created = :attendanceDate');
                $queryBuilder->setParameter('attendanceDate', date('Y-m-d', strtotime(trim($date))).'%');
            }
        }

        ## result in array form.
        $result = $queryBuilder->getQuery()->getScalarResult();

        ## if result has something in it.
        if (is_array($result) && !empty($result)) {
            ## extract the class id from array.
            $result = array_column($result, 'classes_id');
            ## counting the results
            $result = array_count_values($result);
        }

        return $result;
    }


    ## duplication check
    public function duplicationCheck(ParameterBag $bag)
    {
        $queryBuilder = $this->createQueryBuilder('classes');

        if (!empty($bag->get('class_template_id'))) {
            $queryBuilder->andWhere('classes.class_template = :classTemplate')->setParameter(
                'classTemplate',
                $bag->get('class_template_id')
            );
        }

        if (!empty($bag->get('session_id'))) {
            $queryBuilder->andWhere('classes.session = :sessionRef')->setParameter(
                'sessionRef',
                $bag->get('session_id')
            );
        }

        $queryBuilder->andWhere('classes.is_deleted = false');

        return $queryBuilder->getQuery()->getResult();
    }

    ## bulk duplication check
    public function bulkDuplicationCheck(array $classTemplate, int $sesionID)
    {

        foreach ($classTemplate as $key => $value) {
            $bag = new ParameterBag();
            $bag->set('class_template_id', $value);
            $bag->set('session_id', $sesionID);
            $response = $this->duplicationCheck($bag);
            if (!empty($response)) {
                unset($classTemplate[$key]);
            }
        }

        return $classTemplate;
    }


    /**
     * @param bool $parse__YesNoToBoolean
     * @param ParameterBag $bag
     * @return mixed
     * PURPOSE: List of all classes + search feature
     */
    public function listOfClasses(bool $parse__YesNoToBoolean, ParameterBag $bag = null)
    {
        ##  query builder
        $queryBuilder = $this->createQueryBuilder('classes');

        ## left join with to strong connection.
        $queryBuilder->leftJoin('classes.classSubject__', 'classSubject')->addSelect('classSubject');
        $queryBuilder->leftJoin('classes.session', 'session')->addSelect('session');
        $queryBuilder->leftJoin('classes.classFees', 'classFees')->addSelect('classFees');
        $queryBuilder->leftJoin('classes.classFeeSchedules', 'classFeeSchedules')->addSelect('classFeeSchedules');

        if ($bag instanceof ParameterBag) {
            ## search classes of specific session.
            if ($bag->get('session')) {
                $queryBuilder->andWhere('classes.session = :session')->setParameter('session', $bag->get('session'));
            }
            ## search class
            if ($bag->get('class')) {
                $queryBuilder->andWhere('classes.id = :class')->setParameter('class', $bag->get('class'));
            }

        }

        ## ignore delete records
        if (!$parse__YesNoToBoolean) {
            $queryBuilder = $queryBuilder->andWhere('classes.is_deleted = 0');
        }

        return $queryBuilder->orderBy('classes.is_deleted', 'ASC')->getQuery()->getResult();
    }

}
