<?php

namespace App\Controller;

use App\Entity\AllocateTeacherToClassAndSubjects;
use App\Entity\Assesment;
use App\Entity\Classes;
use App\Entity\ClassSubject;
use App\Repository\AllocateTeacherToClassAndSubjectsRepository;
use App\Repository\AttendanceRepository;
use App\Repository\AttendanceStatusRepository;
use App\Repository\UserRepository;
use App\Service\Constants;
use App\Service\DefaultFunction;
use App\Service\FileManagment;
use App\Service\StudentClasses;
use App\Service\StudentSession;
use App\Service\SubjectService;
use App\Service\Teacher\teacherAllocationService;
use Doctrine\ORM\EntityManagerInterface;
use http\Client\Request;
use Knp\Bundle\SnappyBundle\Snappy\Response\PdfResponse;
use Nzo\UrlEncryptorBundle\Annotations\ParamDecryptor;
use phpDocumentor\Reflection\Types\Self_;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\ParameterBag;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;


/**
 * @IsGranted("ROLE_USER")
 * @IsGranted("ROLE_TEACHER")
 * */
class TeacherController extends AbstractController
{

    /**
     * @var UserRepository
     */
    private $user_repository;
    /**
     * @var teacherAllocationService
     */
    private $teacherAllocationService;
    /**
     * @var AttendanceRepository
     */
    private $attendanceRepository;
    /**
     * @var AttendanceStatusRepository
     */
    private $attendanceStatusRepository;
    /**
     * @var EntityManagerInterface
     */
    private $entityManager;

    public function __construct(
        UserRepository $user_repository,
        teacherAllocationService $teacherAllocationService,
        AttendanceStatusRepository $attendanceStatusRepository,
        EntityManagerInterface $entityManager
    ) {
        $this->user_repository = $user_repository;
        $this->teacherAllocationService = $teacherAllocationService;
        $this->attendanceStatusRepository = $attendanceStatusRepository;
        $this->entityManager = $entityManager;
    }

    // Teacher Category Identifier.
    const userCategory = 't';

    /**
     * @Route("/{_locale}/teachers/manage-teacher/", name="teacher_list")
     * ## get the teacher's list
     */
    public function index()
    {
        return $this->render(
            'teacher/index.html.twig',
            [
                'UsersLIst' => $this->user_repository->findBy(['category' => self::userCategory], ['id' => 'DESC']),
            ]
        );
    }

    /**
     * @Route("/{_locale}/teachers/mark-attendance/", name="teacher_mark_attendance")
     * ## get the admin's or other staff list
     * @param \Symfony\Component\HttpFoundation\Request $request
     * @return Response
     */
    public function markAttendance(\Symfony\Component\HttpFoundation\Request $request)
    {
        $bag = new ParameterBag();
        $bag->set('in_date', $request->get('in_date'));
        $bag->set('userCatRef', self::userCategory);

        return $this->render(
            'teacher/markAttendance.html.twig',
            [
                '__user_list' => $this->user_repository->loadAttendanceOfTeacherAndStaff($bag),
                'attendance__status' => $this->attendanceStatusRepository->findBy(['enable' => 1]),
                'userCategory' => self::userCategory,
                'in_date' => $bag->get('in_date'),
            ]
        );
    }


    /**
     * @Route("{_locale}/teacher/requestHandler/remote-request/", name="requestHandler_teacherController")
     * @param \Symfony\Component\HttpFoundation\Request $request
     * @return JsonResponse
     */
    public function ajaxHandler(\Symfony\Component\HttpFoundation\Request $request)
    {
        $response = 'OK';

        ## get the query variable from the get request.
        $query = $request->query->get('query');
        switch ($query) {
            case 'removeTeacherAllocation':
                ## active & de-active request.
                $response = $this->teacherAllocationService->delete_teacher_from_allocation(
                    $request->request->get('rOO')
                );
                break;
            case 'serverTeacherAllocationForm':
                $teacherAllocation = $this->teacherAllocationService->get_teacher_allocation($request->get('rOO'));
                if ($teacherAllocation instanceof AllocateTeacherToClassAndSubjects) {
                    $response = 'escape__fromErro_'.$this->renderView(
                            'teacher/teacherAllocationInputForm.html.twig',
                            ['teacherAllocation' => $teacherAllocation]
                        );
                } else {
                    $response = 'NF';
                }
                break;
            case 'updateTeacherAllocation':
                ## Update teacher allocation
                $response = $this->teacherAllocationService->updateTeacherAllocation($request->request);
                break;
            default:
                $response = 'OK';
                break;
        }

        return new JsonResponse($response, 200);
    }

    /**
     * ==== Reporting
     * */


    /**
     * @Route("{_locale}/teacher/evaluation-report-on-student-base/{teacherRef}/{teacherName}", name="teacherController_evaluationReportOnBaseOfStudent", defaults={"teacherName":"null"})
     * @ParamDecryptor(params={"teacherRef"})
     * @param int $teacherRef
     * @param string $teacherName
     * @param \Symfony\Component\HttpFoundation\Request $request
     * @param AllocateTeacherToClassAndSubjectsRepository $allocateTeacherToClassAndSubjectsRepository
     * @return Response
     */
    public function teacherEvaluationReport(
        int $teacherRef,
        string $teacherName,
        \Symfony\Component\HttpFoundation\Request $request,
        AllocateTeacherToClassAndSubjectsRepository $allocateTeacherToClassAndSubjectsRepository
    ) {
        return $this->render(
            'teacher/reporting/evaluationReport/studentBase.html.twig',
            [
                'teacherRef' => $teacherRef,
                'teacherName' => $teacherName,
                'dateRange' => $request->query->get('date-range'),
                'listOfStudentAllocatedToTeacher' => $allocateTeacherToClassAndSubjectsRepository->findBy(
                    ['teacher' => $teacherRef]
                ),
            ]
        );
    }


    /**
     * @Route("{_locale}/teacher/assign-class-subject/", name="assign_class_subject_to_teacher")
     * @param SubjectService $subjectService
     * @param StudentSession $studentSession
     * @param StudentClasses $studentClasses
     * @param \Symfony\Component\HttpFoundation\Request $request
     * @return Response
     */
    public function assignClassSubjectToTeacher(
        SubjectService $subjectService,
        StudentSession $studentSession,
        StudentClasses $studentClasses,
        \Symfony\Component\HttpFoundation\Request $request
    ) {

        $assignTo = $request->query->get('assign-to') ? $request->query->get('assign-to') : 'class';
        $arrayToView = [
            'classSubjects' => $subjectService->listOfClassSubjects($request->query),
            'sessions' => $studentSession->getAllSessions(),
            'classes_' => $studentClasses->getAllClasses(),
            'assignTo' => $assignTo,
        ];
        if ($assignTo == 'class') {
            $arrayToView['classes'] = $studentClasses->getAllClasses($request->query);
        }

        return $this->render(
            'teacher/assignClassSubjectToTeacher.html.twig',
            $arrayToView
        );
    }

    /**
     * @Route("{_locale}/teacher/assign-class-subject-list/", name="assign_class_subject_to_teacher_list")
     * @param \Symfony\Component\HttpFoundation\Request $request
     * @param StudentSession $studentSession
     * @param StudentClasses $studentClasses
     * @return Response
     */
    public function assignClassSubjectToTeacherList(
        \Symfony\Component\HttpFoundation\Request $request,
        StudentSession $studentSession,
        StudentClasses $studentClasses
    ) {
        return $this->render(
            'teacher/assignedClassSubjectToTeacherList.html.twig',
            [
                'allocationList' => $this->teacherAllocationService->getAllAllocatedSubjectAndClassToTeacher(
                    $request->query
                ),
                'sessions' => $studentSession->getAllSessions(),
                'classes' => $studentClasses->getAllClasses(),
                'teachers' => $this->user_repository->findBy(['category' => self::userCategory], ['id' => 'DESC']),
            ]
        );
    }


    /**
     * ======= Time table
     * */

    /**
     * @Route("{_locale}/teacher/time-table" , name="teacher_time_table")
     *
     * @return PdfResponse|Response
     */
    public function studentTimeTable()
    {
        return $this->render(
            'teacher/timeTable/timetable.html.twig',
            [
                'active_classes' => $this->entityManager->getRepository('App:Classes')->getActiveClassesOfActiveSession(
                    false
                ),
                'teachers' => $this->user_repository->findBy(
                    ['category' => self::userCategory, 'is_active' => 1, 'is_deleted' => 0],
                    ['id' => 'DESC']
                ),

            ]
        );
    }



    /**
     * ================= Assessments
     * */

    /**
     * @Route("{_locale}/student/assessment/teacher-uploads", name="sa_teacher_upload")
     * @param \Symfony\Component\HttpFoundation\Request $request
     * @param FileManagment $fileManagement
     * @param DefaultFunction $defaultFunction
     * @param AllocateTeacherToClassAndSubjectsRepository $allocateTeacherToClassAndSubjectsRepository
     * @return Response
     */
    public function teacher_assessment_uploads(
        \Symfony\Component\HttpFoundation\Request $request,
        FileManagment $fileManagement,
        DefaultFunction $defaultFunction,
        AllocateTeacherToClassAndSubjectsRepository $allocateTeacherToClassAndSubjectsRepository
    ) {
        ## when upload the form
        if ($request->isMethod('POST')) {


            $testPass = true;

            if (empty($request->request->get('all_subjects')) && empty($request->request->get('subject'))) {
                $this->addFlash('assessmentError', 'You should at least one subject or check All Subject Checkbox');
                $testPass = false;
            }

            if (empty($request->request->get('exC__cLs'))) {
                $this->addFlash('assessmentError', 'Class is not selected');
                $testPass = false;
            }
            if (empty($request->request->get('assessment_name'))) {
                $this->addFlash('assessmentError', 'Assessment name is empty');
                $testPass = false;
            }


            if ($testPass) {
                $Assessment = new Assesment();
                $Assessment->setAssesmentName($request->request->get('assessment_name'));
                $Assessment->setDescription($request->request->get('assessment_description'));
                $Assessment->setShowToClassOwnerTeacher(
                    $defaultFunction->parse__boolean($request->request->get('show_to_teachers'))
                );
                $Assessment->setShowToClassSubjectStudent(
                    $defaultFunction->parse__boolean($request->request->get('show_to_students'))
                );
                $Assessment->setShowToStudentGuardians(
                    $defaultFunction->parse__boolean($request->request->get('show_to_parents'))
                );
                $Assessment->setTillValidate(
                    $defaultFunction->convert_datetimeStringIntoObject(($request->request->get('till_validate')))
                );
                $Class = $this->entityManager->getRepository(Classes::class)->find($request->request->get('exC__cLs'));
                $Assessment->setAssesmentClass($Class);
                if ($request->request->get('subject')) {
                    $ClassSubject = $this->entityManager->getRepository(ClassSubject::class)->find(
                        $request->request->get('subject')
                    );
                    $Assessment->setAssessmentSubject($ClassSubject);
                }

                $this->entityManager->persist($Assessment);
                $this->entityManager->flush();

                if (!empty($request->files)) {

                    ## Upload directory
//                    $path = Constants::studentAssessmentPath.DIRECTORY_SEPARATOR.$Assessment->getId(
//                        ).DIRECTORY_SEPARATOR.$Class->getSession()->getid().DIRECTORY_SEPARATOR.$Class->getId();
//
//                    if (!empty($request->request->get('subject'))) {
//                        $path .= DIRECTORY_SEPARATOR.$ClassSubject->getId();
//                    }

                    $path = Constants::studentAssessmentPath.DIRECTORY_SEPARATOR.$Assessment->getId();
                    foreach ($request->files->get('upload_document') as $file) {
                        $fileManagement->uploadFile(
                            [
                                'fileArray' => $file,
                                'path' => $path,
                            ]
                        );
                    }
                }
                $this->addFlash('assessmentSuccess', 'Assessment are saved!');
            }
        }

        $passVariables = [
            'active_classes' => $this->entityManager->getRepository('App:Classes')->getActiveClassesOfActiveSession(
                false
            ),
        ];

        if ($this->getUser()->getCategory() === 't') {
            $passVariables['assignedClassesToTeacher'] = $allocateTeacherToClassAndSubjectsRepository->getUniqueClassAllocationByTeacherID(
                $this->getUser()->getId()
            );
        }

        return $this->render(
            'teacher/assessment/teacherUploads.html.twig',
            $passVariables
        );
    }

    /**
     * @Route("{_locale}/teachers/assessment/list", name="teacher_assessment_list")
     * @return Response
     */
    public function assessmentList()
    {
        $assesments = $this->entityManager->getRepository('App:Assesment')->findAll();

        if ($this->getUser()->getCategory() === 't') {
            $Bag = new ParameterBag();
            $Bag->set(
                'classes',
                $this->entityManager->getRepository(
                    AllocateTeacherToClassAndSubjects::class
                )->getUniqueClassAllocationByTeacherID(
                    $this->getUser()->getId()
                )
            );
            $assesments = $this->entityManager->getRepository(Assesment::class)->advanceSearch($Bag);
        }

        return $this->render(
            'teacher/assessment/assessmentList.twig',
            [
                'assessments' => $assesments,
            ]
        );
    }

}

