<?php

namespace App\Twig;

use App\Chat\ChatOperations;
use App\Entity\Attendance;
use App\Entity\ClassSessionEnrolment;
use App\Entity\ClassSubject;
use App\Entity\User;
use App\Repository\AppSettingsRepository;
use App\Service\AppSettings;
use App\Service\Constants;
use App\Service\DefaultFunction;
use App\Service\FileManagment;
use App\Service\SMSGateways\OperationalionalGatewayClass;
use App\Service\Students\Grade\ClassGradeService;
use App\Service\UserRegistration;
use App\Service\UserService;
use DatePeriod;
use Exception as ExceptionAlias;
use phpDocumentor\Reflection\DocBlock\Tags\Reference\Url;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\HttpFoundation\ParameterBag;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Routing\RequestContext;
use Twig\Extension\AbstractExtension;
use Twig\TwigFilter;
use Twig\TwigFunction;

class
AppExtension extends AbstractExtension
{

    /**
     * @var Request
     */
    private $request;
    /**
     * @var Constants
     */
    private $constants;
    /**
     * @var AppSettingsRepository
     */
    private $app_settings_repository;
    /**
     * @var AppSettings
     */
    private $app_settings_service;
    /**
     * @var ParameterBag
     */
    private $parameter_bag;
    /**
     * @var UserService
     */
    private $user_service;
    /**
     * @var FileManagment
     */
    private $file_managment;
    /**
     * @var UserRegistration
     */
    private $user_registration;
    /**
     * @var DefaultFunction
     */
    private $default_function;
    /**
     * @var ChatOperations
     */
    private $chat_operations;
    /**
     * @var RequestContext|null
     */
    private $requestContext;
    /**
     * @var ClassGradeService
     */
    private $classGradeService;

    /**
     * AppExtension constructor.
     *
     * @param RequestStack $request
     * @param Constants $constants
     * @param AppSettingsRepository $app_settings_repository
     * @param AppSettings $app_settings_service
     * @param ParameterBagInterface $parameter_bag
     * @param UserService $user_service
     * @param FileManagment $file_managment
     * @param UserRegistration $user_registration
     * @param DefaultFunction $default_function
     * @param ChatOperations $chat_operations
     * @param RequestContext|null $requestContext
     * @param ClassGradeService $classGradeService
     */
    public function __construct(
        RequestStack $request,
        Constants $constants,
        AppSettingsRepository $app_settings_repository,
        AppSettings $app_settings_service,
        ParameterBagInterface $parameter_bag,
        UserService $user_service,
        FileManagment $file_managment,
        UserRegistration $user_registration,
        DefaultFunction $default_function,
        ChatOperations $chat_operations,
        ClassGradeService $classGradeService,
        RequestContext $requestContext = null
    ) {
        $this->request = $request;
        $this->constants = $constants;
        $this->app_settings_repository = $app_settings_repository;
        $this->app_settings_service = $app_settings_service;
        $this->parameter_bag = $parameter_bag;
        $this->user_service = $user_service;
        $this->file_managment = $file_managment;
        $this->user_registration = $user_registration;
        $this->default_function = $default_function;
        $this->chat_operations = $chat_operations;
        $this->requestContext = $requestContext;
        $this->classGradeService = $classGradeService;
    }


    /**
     * @return array
     */
    public
    function getFilters(): array
    {
        return [
            // If your filter generates SAFE HTML, you should add a third
            // parameter: ['is_safe' => ['html']]
            // Reference: https://twig.symfony.com/doc/2.x/advanced.html#automatic-escaping
            new TwigFilter('appSettings', [$this, 'appSettingsValue']),
            new TwigFilter('baseName', [$this, 'baseName']),
        ];
    }

    /**
     * @return array
     */
    public
    function getFunctions(): array
    {
        return [
            new TwigFunction('app_secret', 'getenv'),
            new TwigFunction('yesNo', [$this, 'returnYesNo']),
            new TwigFunction('c_locale', [$this, 'currentLocale']),
            new TwigFunction('supportedLang', [$this, 'supportedLang']),
            new TwigFunction('enabledLang', [$this, 'getUserEnabledLanguages']),
            new TwigFunction('appSettingUniqueNames', [$this, 'getAppSettingsUniqueFieldsValue']),
            new TwigFunction('is_translation_enabled', [$this, 'is_translation_enabled']),
            new TwigFunction('array_intersect', [$this, 'compare_two_arrays']),
            new TwigFunction('menu_list', [$this, 'getMenuList']),
            new TwigFunction('in_array', [$this, 'in_array']),
            new TwigFunction('refineArguments', [$this, 'refineArguments']),
            new TwigFunction('fekaraCheckList', [$this, 'CheckList']),
            new TwigFunction('is_checkListMissing', [$this, 'is_checkListMissing']),
            new TwigFunction('dateRangeBetweenTwoDates', [$this, 'listOfDatesBetweenTwoDatesInterval']),
            new TwigFunction('absolute_url', [$this, 'generateAbsoluteUrl']),
            new TwigFunction('getEnv', [$this, 'getEnv']),
            ## instance of
            new TwigFunction('isInstanceof', [$this, 'isInstanceof']),
            ## get app settings from the sessions
            new TwigFunction('getAppSetting', [$this, 'getAppSettingNode']),
            ## get app settings from database
            new TwigFunction('getFreshAppSetting', [$this, 'getFreshAppSettingNode']),
            ## encode and decode the string
            new TwigFunction('secure', [$this, 'encoder_decoder']),
            ## formatize date
            new TwigFunction('date_format', [$this, 'date_format']),
            new TwigFunction('datetime_format', [$this, 'datetime_format']),
            ## formatize time
            new TwigFunction('time_format', [$this, 'time_format']),
            ## get user's full category by giving the short name of category
            new TwigFunction('get_the_user__full_category', [$this, 'get_the_user__full_category']),

            ## load chat attachments.
            new TwigFunction('loadChatAttachments', [$this, 'loadChatAttachments']),

            ## get human readable time
            new TwigFunction('ago__', [$this, 'time_elapsed_string']),

            ## get the array column
            new TwigFunction('array__column', [$this, 'array_column_']),
            new TwigFunction('implode__', [$this, 'implode_']),
            new TwigFunction('convertBasePathIntoUrl', [$this, 'convertBasePathIntoUrl']),
            new TwigFunction('user_avatar', [$this, 'user_avatar']),
            new TwigFunction('applicationSupportedRoles', [$this, 'getSupportedUserRoles']),
            new TwigFunction('parseCf', [$this, 'parseCustomFieldData']),
            new TwigFunction('getUploadedFile', [$this, 'getUploadedFile']),
            ## Calculate grade
            new TwigFunction('studentGradeCalculator', [$this, 'calculateGradeFromMarksPercentage']),
            new TwigFunction('twig_json_decode', [$this, 'twig_json_decode']),
        ];
    }

    /*
     * ============== FUNCTIONS LIST
     * */


    public function twig_json_decode($string)
    {
        return json_decode($string);
    }

    public
    function getEnv(
        $variableName
    ) {
        return getenv($variableName);
    }

    /**
     * @param $classId
     * @param $MarksPercentage
     * @return mixed|string|null
     * Purpose: Calculate percentage on class base
     */
    public
    function calculateGradeFromMarksPercentage(
        $classId,
        $MarksPercentage
    ) {
        return $this->classGradeService->calculateGradeFromMarksPercentage($classId, $MarksPercentage);
    }

    /**
     * Returns the absolute URL for the given absolute or relative path.
     *
     * This method returns the path unchanged if no request is available.
     *
     * @param string $path The path
     *
     * @return string The absolute URL
     *
     * @see Request::getUriForPath()
     */
    public
    function generateAbsoluteUrl(
        $path
    ) {
        if (false !== strpos($path, '://') || '//' === substr($path, 0, 2)) {
            return $path;
        }
        if (!$request = $this->request->getMasterRequest()) {
            if (null !== $this->request && '' !== $host = $this->request->getHost()) {
                $scheme = $this->request->getScheme();
                $port = '';
                if ('http' === $scheme && 80 != $this->requestContext->getHttpPort()) {
                    $port = ':'.$this->requestContext->getHttpPort();
                } elseif ('https' === $scheme && 443 != $this->requestContext->getHttpsPort()) {
                    $port = ':'.$this->requestContext->getHttpsPort();
                }
                if ('#' === $path[0]) {
                    $queryString = $this->request->getQueryString();
                    $path = $this->request->getPathInfo().($queryString ? '?'.$queryString : '').$path;
                } elseif ('?' === $path[0]) {
                    $path = $this->request->getPathInfo().$path;
                }
                if ('/' !== $path[0]) {
                    $path = rtrim($this->request->getBaseUrl(), '/').'/'.$path;
                }

                return $scheme.'://'.$host.$port.$path;
            }

            return $path;
        }
        if ('#' === $path[0]) {
            $path = $request->getRequestUri().$path;
        } elseif ('?' === $path[0]) {
            $path = $request->getPathInfo().$path;
        }
        if (!$path || '/' !== $path[0]) {
            $prefix = $request->getPathInfo();
            $last = \strlen($prefix) - 1;
            if ($last !== $pos = strrpos($prefix, '/')) {
                $prefix = substr($prefix, 0, $pos).'/';
            }

            return $request->getUriForPath($prefix.$path);
        }

        return $request->getSchemeAndHttpHost().$path;
    }

    public
    function CheckList()
    {
        return $this->app_settings_service->fekaraCheckList();
    }

    public
    function is_checkListMissing()
    {
        $response = false;
        $CheckList = $this->CheckList();
        foreach ($CheckList as $value) {
            if (!$value) {
                $response = true;
                break;
            }
        }

        return $response;
    }

## instance of
    public
    function isInstanceof(
        $var,
        $instance
    ) {

        switch ($instance) {
            case  'ClassSessionEnrolment':
                $instance = new ClassSessionEnrolment();
                break;
            case 'user':
                $instance = new User();
                break;
            case 'attendance':
                $instance = new Attendance();

                break;
            case 'classSubject':
                $instance = new ClassSubject();
                break;
        }

        return $var instanceof $instance;
    }

    public
    function array_column_(
        $key,
        $array
    ) {
        return array_column($array, $key);
    }

    public
    function getFreshAppSettingNode(
        $key,
        $onlyOne = 'a'
    ) {
        ## get the real setting name
        $key = $this->getAppSettingsUniqueFieldsValue($key);
        $response = $this->app_settings_service->getDataFromAppSettings($key, $onlyOne);
        if ($key == 's_logo' && empty($response)) {
            $response = Constants::FekaraLogoPath;
        }

        return $response;
    }

    /**
     * @param string|null $string
     * @param bool|null $encode
     *
     *  Encode or decode the string for security purpose.
     *
     * @return bool|string
     */
    public
    function encoder_decoder(
        $string,
        ?bool $encode = true
    ) {
        if (is_string($string)) {
            return $this->default_function->encoder_decoder($string, $encode);
        }
    }

    /**
     * @param $value
     * This functions will take null, int or string or check, if given value is empty or not, if not empty then return
     * yes neither return no.
     *
     * @return string
     */
    public
    function returnYesNo(
        $value
    ) {
        $returnValue = 'No';
        if ($value) {
            $returnValue = 'Yes';
        }

        return $returnValue;
    }

    /*
     * return the datetime object with the default date format which is selected in the app settings.
     * */
    public
    function date_format(
        $dateObject
    ) {

        if (empty($dateObject)) {
            return '';
        }

        /** @var \DateTimeInterface $dateObject */
        if (!empty($this->getAppSettingNode('date_format'))) {
            return $dateObject->format($this->getAppSettingNode('date_format'));
        } else {
            return $dateObject->format('d-M-Y');
        }

    }

    /*
     * return the datetime object with the default date format which is selected in the app settings.
     * */
    public
    function time_format(
        $dateObject
    ) {

        if (empty($dateObject)) {
            return '';
        }
        /** @var \DateTimeInterface $dateObject */
        if (!empty($this->getAppSettingNode('time_format'))) {
            return $dateObject->format($this->getAppSettingNode('time_format'));
        } else {
            return $dateObject->format('h:i');
        }

    }


    /*
     * return the datetime object with the default date format which is selected in the app settings.
     * */
    public
    function datetime_format(
        $dateObject
    ) {

        if (empty($dateObject)) {
            return '';
        }


        /** @var \DateTimeInterface $dateObject */
        if (!empty($this->getAppSettingNode('time_format')) && !empty($this->getAppSettingNode('date_format'))) {
            return $dateObject->format($this->getAppSettingNode('date_format')).' '.$dateObject->format(
                    $this->getAppSettingNode('time_format')
                );
        } else {
            return $dateObject->format('d-M-Y').' '.$dateObject->format('h:i');
        }

    }


    /**
     * Get the current locale
     * */
    public
    function currentLocale()
    {
        return $this->request->getCurrentRequest()->getSession()->get('_locale', 'en');
    }

    /**
     * @param null $key
     * Application supported languages.
     *
     * @return string
     */
    public
    function supportedLang(
        $key = null
    ) {
        $response = '';
        if (!empty($key)) {
            if (isset($this->constants::SupportedLanguages[$key])) {
                $response = $this->constants::SupportedLanguages[$key];
            } else {
                $response = 'No, such kind of langauge exits.';
            }

        } else {
            $response = $this->constants::SupportedLanguages;
        }

        return $response;
    }

    /**
     * App Settings Related.
     * Check is translation is enabled.
     *  Replace this method with the get AppSettingNode method
     *
     * @return string
     */
    public
    function is_translation_enabled()
    {
        return $this->app_settings_service->getAppSettingsFromSessions('is_multilingual');
        /*$response    = $this->app_settings_service->getUserPreferLocal();
        $returnValue = $response[ $this->constants::AppSettingsLabels['is_multilingual'] ];

        ## is multilingual enabled
        if ( empty( $response[ $this->constants::AppSettingsLabels['is_multilingual'] ] ) ) {
            $returnValue = 'off';
        }

        return $returnValue;*/
    }

    /*
     * Get the supported languages list
     * PURPOSE: I've to get all languages list when each template is render, so, I've created this so, it ease to use in each requested twig template.
     * Replce this method with the getAppSettingNode Method
     * @return array
     * */
    public
    function getUserEnabledLanguages()
    {
        $returnValue = [];
        ## if multilingual option is enabled then get the user preference languages.
        if ($this->is_translation_enabled() == 'on') {
            ## get enabled locales
            $returnValue = $this->app_settings_service->getAppSettingsFromSessions('enabled_Languages');

            if (empty($returnValue)) {
                ## if response is empty and multilingual is enable then return app supported languages.
                $returnValue = $this->constants::SupportedLanguages;
            } elseif (is_array($returnValue) && !empty($returnValue)) {
                ## if enabled language variable is not empty then mapped into supported lang array
                ## if application selected multiple languages..
                $temp_ar = [];
                foreach ($returnValue as $key => $value) {
                    $temp_ar[$value] = $this->constants::SupportedLanguages[$value];
                }
                $returnValue = $temp_ar;
            } elseif (is_string($returnValue) && !empty($returnValue)) {
                ## if 1 language is selected from settings..
                $returnValue = $this->constants::SupportedLanguages[$returnValue];
            }
        }


        return $returnValue;
    }

    /**
     * This method is used to get the app settings, node .
     * ex. if you want to get per page length then pass the record_per_page which is defined in the constant.php file.
     *
     * @param string|null $node
     *
     * @return mixed|string
     */
    public
    function getAppSettingNode(
        ?string $node = null
    ) {
        return $this->app_settings_service->getAppSettingsFromSessions($node);
    }

    /**
     * @param string|null $key
     * PURPOSE: get the setting name from app_settings table.
     *
     * @return string
     */
    public
    function getAppSettingsUniqueFieldsValue(
        ?string $key
    ) {
        $returnValue = '';
        if (isset($this->constants::AppSettingsLabels[$key])) {
            $returnValue = $this->constants::AppSettingsLabels[$key];
        }

        return $returnValue;
    }

    /**
     * Twig has no in array function for its array key, so I just add PHP function in it.
     *
     * @param $needle
     * @param $hackstack
     *
     * @return bool
     */
    public
    function in_array(
        $needle,
        $hackstack
    ) {
        return in_array($needle, $hackstack);
    }

    /*
     * Return Menu array based on the user permissions.
     * */
    public
    function getMenuList(
        $userTye = 'a'
    ) {
        $menuArray = $this->constants::MenuArray;

        if ($userTye == 'g') {
            $menuArray = $this->constants::GuardianMenuArray;
        } elseif ($userTye == 's') {
            $menuArray = $this->constants::StudentMenuArray;
        } elseif ($userTye == 't') {
            $menuArray = $this->constants::TeacherMenuArray;
        }

        return $menuArray;
    }

    /*
     * in_array search needle in a array, if we needle is array then use this function.
     * */
    /**
     * @param $ar1
     * @param $ar2
     *
     * @return array
     */
    public
    function compare_two_arrays(
        $ar1,
        $ar2
    ) {
        return array_intersect($ar1, $ar2);
    }

    /**
     * @param        $array
     * @param string $locale
     *
     * @return mixed
     */
    public
    function refineArguments(
        $array,
        $locale = 'en'
    ) {
        $array['_locale'] = $locale;

        return $array;
    }

    /*
     * User avatar
     * */
    public
    function user_avatar(
        $UserId,
        ?bool $returnHtml = true,
        $userCategory = 's'
    ) {
        ## if user id is empty then return the dummy image.
        if (empty($UserId)) {
            return $response = $this->constants::UserAvatardummyImage;
        }

        return $this->user_service->getUserAvatar($UserId, $returnHtml, $userCategory);
    }

    /*
     * Get user's roles.
     * */
    public
    function getSupportedUserRoles()
    {
        return $this->constants::UserRoles;
    }


    /*
     * Parse custom field option data.
     * */
    public
    function parseCustomFieldData(
        $variable,
        $typeOfField
    ) {
        return $this->user_registration->parseCustomFieldData($variable, $typeOfField);
    }

    /**
     * @param bool $public
     * @param      $path
     * @param bool $need_url
     * @param bool $rootPath
     * @param bool $returnSingleIfOne
     * @param bool $findInDirectory
     * @param null $searchPattern
     * @param null $absoulteUrl
     *
     * @return array
     * PURPOSE: Get uploaded files.
     */
    public
    function getUploadedFile(
        $public = true,
        $path = null,
        $need_url = true,
        $rootPath = false,
        $returnSingleIfOne = true,
        $findInDirectory = false,
        $searchPattern = null,
        $absoluteUrl = null,
        $folderNameOrFileName = null
    ) {
        $temp_ar = [
            'need_url' => $need_url,
            'path' => $path,
            'public' => $public,
            'projectPath' => $rootPath,
            'findInDirectory' => $findInDirectory,
            'finderPattern' => $searchPattern,
            'absPath' => $absoluteUrl,
            'need_only_file_or_folder_name' => $folderNameOrFileName,
        ];

        $files = $this->file_managment->getUploadedFile($temp_ar);

        ## if we've single file then return the single file.
        if (count($files) == 1 && $returnSingleIfOne) {
            $files = $files[0];
        }

        ## if we've multiple files then return multiple files.
        return $files;
    }


    public
    function get_the_user__full_category(
        string $category
    ) {
        $response = '';


        switch ($category) {
            ## if category is empty
            case empty($category):
                break;
            ## category is admin
            case 'a':
                $response = 'admin';
                break;
            ## user is student
            case 's':
                $response = 'student';
                break;
            ## user is teacher
            case 't':
                $response = 'teacher';
                break;
            ## user is guardin.
            default:
                $response = '';
                break;
        }


        return $response;

    }

## load chat attachments.
    public
    function loadChatAttachments(
        $date,
        $message_ref
    ) {
        return $this->chat_operations->load___chat_message_files($date, $message_ref);
    }

## get the human readable time
    public
    function time_elapsed_string(
        $datetime,
        $full = false
    ) {

        try {
            $now = new \DateTime();
            $ago = new \DateTime($datetime);
        } catch (ExceptionAlias $e) {
            return $e->getMessage();
        }


        $diff = $now->diff($ago);
        $diff->w = floor($diff->d / 7);
        $diff->d -= $diff->w * 7;

        $string = array(
            'y' => 'year',
            'm' => 'month',
            'w' => 'week',
            'd' => 'day',
            'h' => 'hour',
            'i' => 'minute',
            's' => 'second',
        );
        foreach ($string as $k => &$v) {
            if ($diff->$k) {
                $v = $diff->$k.' '.$v.($diff->$k > 1 ? 's' : '');
            } else {
                unset($string[$k]);
            }
        }

        if (!$full) {
            $string = array_slice($string, 0, 1);
        }

        return $string ? implode(', ', $string).' ago' : 'just now';
    }


    /*
     *
     * ========================== Twig Filters.
     *
     * */

    public
    function appSettingsValue(
        $argu = null
    ) {
        /** @var ParameterBag */
        return $argu->request;
    }

    /**
     * covert base path into the url.
     *
     * @param $basePath
     *
     * @return string
     */
    public
    function convertBasePathIntoUrl(
        $basePath
    ) {
        return $this->file_managment->convertBasePathIntoUrl($basePath);
    }

## return the file base name
    public
    function baseName(
        string $string
    ) {
        return basename($string);
    }


    /**
     *  return the list of date available in two dates interval
     *
     * @param $dateInterval
     *
     * @return DatePeriod|string|null
     * @throws ExceptionAlias
     */
    public
    function listOfDatesBetweenTwoDatesInterval(
        $dateInterval
    ) {
        try {
            return $this->default_function->listOfDatesBetweenTwoDatesInterval($dateInterval);
        } catch (ExceptionAlias $exception) {
            return $exception->getMessage();
        }
    }

    public function implode_($array, $pieces = ',')
    {
        return implode($pieces, $array);
    }

}

