<?php


namespace App\Service;


use App\Controller\AdminController;
use App\Controller\FileServerController;
use App\Controller\StudentController;
use App\Controller\TeacherController;
use App\Entity\CustomFields;
use App\Entity\Guardian;
use App\Entity\User;
use App\Entity\WhoCanUpdateCustomField;
use App\Repository\CustomFieldsRepository;
use App\Repository\GuardianRepository;
use App\Repository\UserRepository;
use App\Repository\WhoCanUpdateCustomFieldRepository;
use App\Service\Guardian\GuardianService;
use Doctrine\ORM\EntityManagerInterface;
use Exception;
use phpDocumentor\Reflection\Types\Resource_;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\ParameterBag;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Security\Core\Encoder\PasswordEncoderInterface;
use Symfony\Component\Security\Core\Encoder\UserPasswordEncoderInterface;
use Symfony\Component\Security\Core\Exception\InvalidCsrfTokenException;
use Symfony\Component\Security\Csrf\CsrfToken;
use Symfony\Component\Security\Csrf\CsrfTokenManagerInterface;

class UserService
{

    /**
     * @var UserRepository
     */
    private $user_repository;
    /**
     * @var EntityManagerInterface
     */
    private $entity_manager;
    /**
     * @var DefaultFunction
     */
    private $default_function;
    /**
     * @var FileManagment
     */
    private $file_managment;
    private $errorArray = [];
    /**
     * @var PasswordEncoderInterface
     */
    private $password_encoder;
    /**
     * @var UrlGeneratorInterface
     */
    private $url_generator;
    /**
     * @var Constants
     */
    private $constants;
    /**
     * @var UserRegistration
     */
    private $user_registration;
    /**
     * @var CustomFields
     */
    private $custom_fields;
    /**
     * @var CustomFieldsRepository
     */
    private $custom_fields_repository;
    /**
     * @var WhoCanUpdateCustomFieldRepository
     */
    private $who_can_update_custom_field_repository;
    /**
     * @var TriggerNotifications
     */
    private $trigger_notifications;
    /**
     * @var CsrfTokenManagerInterface
     */
    private $csrf_token_manager;
    /**
     * @var CustomTokenManager
     */
    private $custom_token_manager;
    /**
     * @var GuardianService
     */
    private $guardian_service;


    /**
     * UserService constructor.
     *
     * @param GuardianService $guardian_service
     * @param CustomTokenManager $custom_token_manager
     * @param CsrfTokenManagerInterface $csrf_token_manager
     * @param UserRepository $user_repository
     * @param EntityManagerInterface $entity_manager
     * @param DefaultFunction $default_function
     * @param FileManagment $file_managment
     * @param UserPasswordEncoderInterface $password_encoder
     * @param UrlGeneratorInterface $url_generator
     * @param Constants $constants
     * @param UserRegistration $user_registration
     * @param CustomFieldsRepository $custom_fields_repository
     * @param WhoCanUpdateCustomFieldRepository $who_can_update_custom_field_repository
     * @param TriggerNotifications $trigger_notifications
     */
    public function __construct(
        GuardianService $guardian_service,
        CustomTokenManager $custom_token_manager,
        CsrfTokenManagerInterface $csrf_token_manager,
        UserRepository $user_repository,
        EntityManagerInterface $entity_manager,
        DefaultFunction $default_function,
        FileManagment $file_managment,
        UserPasswordEncoderInterface $password_encoder,
        UrlGeneratorInterface $url_generator,
        Constants $constants,
        UserRegistration $user_registration,
        CustomFieldsRepository $custom_fields_repository,
        WhoCanUpdateCustomFieldRepository $who_can_update_custom_field_repository,
        TriggerNotifications $trigger_notifications
    ) {
        $this->user_repository = $user_repository;
        $this->entity_manager = $entity_manager;
        $this->default_function = $default_function;
        $this->file_managment = $file_managment;
        $this->password_encoder = $password_encoder;
        $this->url_generator = $url_generator;
        $this->constants = $constants;
        $this->user_registration = $user_registration;
        $this->custom_fields_repository = $custom_fields_repository;
        $this->who_can_update_custom_field_repository = $who_can_update_custom_field_repository;
        $this->trigger_notifications = $trigger_notifications;
        $this->csrf_token_manager = $csrf_token_manager;
        $this->custom_token_manager = $custom_token_manager;
        $this->guardian_service = $guardian_service;
    }

    /**
     * @return string
     * @var ParameterBag $requestData
     */
    public function activeDeactiveUser($requestData)
    {
        $returnArray = '';

        ## get the user reference.
        $userReference = $requestData->get('userReference');

        ## if user reference is not found then
        if (empty($userReference)) {
            $returnArray = 'User reference not found';
        } else {
            ## if we found user reference
            $userObject = $this->user_repository->find($requestData->get('userReference'));
            $userObject->setIsActive(!$userObject->getIsActive());
            try {
                $this->entity_manager->persist($userObject);
                $this->entity_manager->flush();
                $returnArray = 'OK';
            } catch (\Exception $e) {
                $returnArray = $e->getMessage();
            }

        }

        return $returnArray;

    }


    /**
     * @param ParameterBag $request
     *
     * @param ParameterBag|null $requestUploadFile
     *
     * @return array
     */
    public function updateProfile($request, ParameterBag $requestUploadFile = null)
    {
        $response = [];

        ## get the user
        $User = $this->pre__userCheck($request);
        $admissionNumber = null;

        if (!$User instanceof User) {
            ## if user instance is not found., it mean we are adding a new student.
            ## when we are doing we wil keep the password by yourself & make an account.
            $User = new User();
            $User->setPassword('testing');
            $User->setCategory($request->get('category', 's'));
            $User->setIsActive(true);
            ## set user role to the student
            $User->setRoles($this->user_registration->setRoleAccordingToUserCategory($request->get('category', 's')));
        }

        ## validation.
        if (empty($request->get('fName'))) {
            $response = $this->default_function->push_error($response, 'First name is missing');
        }

        ## validation.
        if (empty($request->get('lname'))) {
            $response = $this->default_function->push_error($response, 'Last name is missing');
        }

        ## validation.
        if (empty($request->get('gender'))) {
            $response = $this->default_function->push_error($response, 'Please select Gender');
        }

        ## if email is empty then sent back
        if (empty($request->get('email'))) {
            $response = $this->default_function->push_error($response, 'Email is required');
        }

        if (!empty($response)) {
            return $response;
        }

        ## email validation - check if email of this is user is already exits or not.
        if (!empty($request->get('email')) && (!filter_var($request->get('email'), FILTER_VALIDATE_EMAIL))) {
            ##push errors in the array
            $response = $this->default_function->push_error(
                $response,
                $request->get('email').' ,  Invalid email address'
            );
        }

        ## email already existance check
        if (!empty($this->user_repository->getAssociatedUserWithEmail($request->get('email'), $User->getId()))) {
            $response = $this->default_function->push_error(
                $response,
                $request->get('email').' , is already register with the other user'
            );
        }


        ## update guardian if ask for.
        if (!empty($request->get('guard__'))) {
            try {
                /** @var   Guardian $Guardian */
                $Guardian = $this->guardian_service->getGuardian($request->get('guard__'));
                if ($Guardian instanceof Guardian) {
                    $User->addGuardians($Guardian);
                } else {
                    $response = $this->default_function->push_error($response, $Guardian);
                }
            } catch (Exception $exception) {
                $response = $this->default_function->push_error($response, $exception->getMessage());
            }
        }


        ## return error if we've it.
        if (!empty($response)) {
            return $response;
        }

        ## set user f and l name, email and gender.
        $User->setFName($request->get('fName'));
        $User->setLName($request->get('lname'));
        $User->setEmail($request->get('email'));
        $User->setGender($request->get('gender'));

        // if file array is not empty
        if (!empty($requestUploadFile->get('profile_avatar'))) {
            $response_ = $this->uploadUserProfile($requestUploadFile->get('profile_avatar'), $request->get('id'));
            if ($response_ <> 'OK' || (is_array(
                        $response_
                    ) && !empty($response_['response']) && $response_['response'] <> 'OK')) {
                $response = $this->default_function->push_error($response, $response_['response']);
            }
        }

        ## return error if we've it.
        if (!empty($response)) {
            return $response;
        }

        ## set admission number if exits
        if (!empty($request->get('__n_id'))) {
            $User->setAdmissionNumber($request->get('__n_id'));
        }

        try {
            ## save user in a database.
            $this->entity_manager->persist($User);
            $this->entity_manager->flush();

            ## after saving user add his admission number if is empty.
            if (empty($User->getAdmissionNumber())) {
                $User->setAdmissionNumber($User->getId());
                $this->entity_manager->persist($User);
                $this->entity_manager->flush();
            }

        } catch (\Exception $exception) {
            $response = $this->default_function->push_error($response, $exception->getMessage());
        }

        ## if we've no errors then return
        if (empty($response)) {
            $response = 'OK'.$User->getId();
        }

        return $response;

    }

    /**
     * Upload user profile
     *
     * @param ParameterBag $requestFileArray
     * @param              $userId
     *
     * @param string $userCategory
     *
     * @return array
     */
    public function uploadUserProfile($requestFileArray, $userId, $userCategory = 's')
    {
        $temp = [
            'fileArray' => $requestFileArray,
            'allowedExtension' => ['jpg', 'jpeg', 'png', 'gif'],
            'public' => true,
            'path' => $this->constants::UserProfilePath.$userId.DIRECTORY_SEPARATOR,
            'deleteFolder' => true,
            'returnPath' => true,
        ];

        if ($userCategory === 'g') {
            ## guardian images are stored in a different folders.
            $temp['path'] = $this->constants::guardianProfilePath.$userId.DIRECTORY_SEPARATOR;
        }

        return $this->file_managment->uploadFile($temp);
    }

    /**
     * Get user avatar
     *
     * @param        $UserId
     * @param        $returnHtml
     * @param string $userCategory
     *
     * @return array|mixed|string
     */
    public function getUserAvatar($UserId, $returnHtml, $userCategory = 's')
    {
        $temp = [
            'public' => true,
            'path' => $this->constants::UserProfilePath.$UserId,
            'need_url' => true,
        ];

        if ($userCategory === 'g') {
            ## guardian images are stored in a different folders.
            $temp['path'] = $this->constants::guardianProfilePath.$UserId;
        }
        $response = $this->file_managment->getUploadedFile($temp);
        ## if response is not empty and this is an array.
        if (!empty($response) && is_array($response)) {
            $response = $response[0];
            if ($returnHtml) {
                $response = '<span class="kt-badge kt-badge--username kt-badge--unified-success kt-badge--lg kt-badge--rounded kt-badge--bold">S</span>';
            }
        } else {
            $response = $this->constants::UserAvatardummyImage;
            if ($returnHtml) {
                $response = '<span class="kt-badge kt-badge--username kt-badge--unified-success kt-badge--lg kt-badge--rounded kt-badge--bold">S</span>';
            }
        }
        return $response;
    }

    /**
     * @param ParameterBag $request
     *
     * @param null $User
     *
     * @return array|bool|mixed
     * PURPOSE: Update user password.
     */
    public function updateUserPassword(ParameterBag $request, $User = null)
    {
        $response = [];

        ## get the user
        ## if user id is missing then return with the error.
        ## If user not found in the database.
        if (!$User instanceof User) {
            $User = $this->pre__userCheck($request);
            if (!$User instanceof User) {
                $response = 'Invalid User details';
            }
        }

        ## return error if we've it.
        if (!empty($response)) {
            return $response;
        }

        ## don't ask current password.
        if (empty($request->get('ask___cp', null))) {
            if (empty($request->get('currentPassword')) && $request->get('____b') <> '__') {
                $response = $this->default_function->push_error($response, 'Please type your current password');
            }
        }

        ## validation.
        if (empty($request->get('newPassword'))) {
            $response = $this->default_function->push_error($response, 'Please type new password');
        }

        ## validation.
        if (empty($request->get('verifyPassword')) && $request->get('____b') <> '__') {
            $response = $this->default_function->push_error(
                $response,
                'Please verify your password, by retyping in the new password. '
            );
        }


        ## return error if we've it.
        if (!empty($response)) {
            return $response;
        }

        ## validation, new password and retype password is not matched.
        if ($request->get('newPassword') <> $request->get('verifyPassword') && $request->get('____b') <> '__') {
            $response = $this->default_function->push_error(
                $response,
                'Your new password and verify password is not matched'
            );
        }

        ## return error if we've it.
        if (!empty($response)) {
            return $response;
        }

        ## check given password is validate.
        ## if password is not matched.
        ## don't ask current password.
        if (empty($request->get('ask___cp', null))) {
            ## don't check current password
            if (!$this->password_encoder->isPasswordValid($User, $request->get('currentPassword')) && $request->get(
                    '____b'
                ) <> '__') {
                $response = $this->default_function->push_error(
                    $response,
                    'Your current password is not matched, please type correct password. '
                );
            }
        }

        ## return error if we've it.
        if (!empty($response)) {
            return $response;
        }

        ## if every thing goes well & password is matched.
        ## update the password
        $User->setPassword($this->password_encoder->encodePassword($User, $request->get('newPassword')));
        try {
            $this->entity_manager->persist($User);
            $this->entity_manager->flush();
        } catch (\Exception $exception) {
            $response = $this->default_function->push_error($response, $exception->getMessage());
        }

        ## return response
        if (!empty($response)) {
            return $response;
        } else {

            ## send notification to user, password update.
            $temp_ar = [
                'event_name' => 'PASSWORD_UPDATE',
                'user' => $User,
            ];
            $this->trigger_notifications->triggerNotifications($temp_ar);


            return 'OK';
        }

    }

    /**
     * Update user's account info
     *
     * @param ParameterBag $request
     *
     * @return array|mixed|string
     */
    public function updateAccountInfo(ParameterBag $request)
    {
        $response = [];

        ## get the user
        $User = $this->pre__userCheck($request);

        if (!$User instanceof User) {
            $response = 'Invalid User details';
        }

        ## return error if we've it.
        if (!empty($response)) {
            return $response;
        }


        ## after found ther user in the database, check the given admission number is added by other user or not.
        ## check roll no or identification number is already in the use of other user or not.
        ## find the user with the roll no or identification no.
        try {
            $response = $this->user_repository->findNotOneBy(
                'admission_number',
                $request->get('__n_id'),
                $User->getId()
            );
        } catch (\Exception $exception) {
            $response = $exception->getMessage();
        }

        if (!empty($response)) {
            ## yes, this admission number is assigend to other user
            return sprintf('This admission number or identification number is already assign to other user');
        }

        ## validation check
        if (!is_array($request->get('roles'))) {
            return 'Invalid Roles';
        }


        try {
            ##update user role.
            $User->setRoles($request->get('roles'));
            $User->setAdmissionNumber($request->get('__n_id'));
            $this->entity_manager->persist($User);
            $this->entity_manager->flush();
            $response = 'OK';
        } catch (\Exception $exception) {
            $response = $this->default_function->push_error($response, $exception->getMessage());
        }

        return $response;
    }


    /**
     *  update data which is stored in the custom field.
     *
     * @param Request $request
     *
     * @return User|array|mixed|string|null
     */
    public function updateOtherInfo(Request $request)
    {
        ## get the user based on the user id.
        $response = $this->pre__userCheck($request->request);
        if ($response instanceof User) {
            ## remove id parameter
            $request->request->remove('id');
            ## validate & save the customer's data.
            $response = $this->user_registration->validateAndUpdateCustomFieldData_part1(
                $request->request,
                $request,
                $response
            );
            if (empty($response)) {
                return $response = 'OK';
            } else {
                return $response;
            }
        } else {
            return 'Invalid User details';
        }
    }

    /**
     * @param ParameterBag $request
     *
     * Lock and Unlock Custom field for the user.
     *
     * @return User|array|mixed|null
     */
    public function lockField(ParameterBag $request)
    {
        $backResponse = '';

        ## find the user against the user id.
        $User = $this->pre__userCheck($request);

        if ($User instanceof User) {
            ## if we found the user

            ## get the custom field reference, which we've to lock
            try {
                $CustomField = $this->custom_fields_repository->find($request->get('FRef'));
                if ($CustomField instanceof CustomFields) {
                    ## if we found the custom field.


                    if ($request->get('lockfRf')) {
                        ## remove the locking by deleting the custom field.
                        $whocanUpdate = $this->who_can_update_custom_field_repository->find($request->get('lockfRf'));
                        if ($whocanUpdate instanceof WhoCanUpdateCustomField) {
                            $this->entity_manager->remove($whocanUpdate);
                        }
                    } else {
                        ## enable locking.
                        $whocanUpdate = new WhoCanUpdateCustomField ();
                        $whocanUpdate->addCustomFieldRef($CustomField);
                        $whocanUpdate->addUserReference($User);
                        ## save the data
                        $this->entity_manager->persist($whocanUpdate);
                    }
                    $this->entity_manager->flush();

                    ## send success response.
                    $backResponse = 'OK';

                } else {
                    ## report error - if we didn't found custom field against the given id.
                    $backResponse = $CustomField;
                }
            } catch (\Exception $exception) {
                $backResponse = $exception->getMessage();
            }

        } else {
            ## if we didn't find the user then return the error.
            $backResponse = $User;
        }

        return $backResponse;
    }


    /**
     * @param $request
     * This function, check the request parameter & validate the given user is in database or not.
     *
     * @return User|array|mixed|null
     */
    public function pre__userCheck(ParameterBag $request)
    {
        $response = [];
        ## if user id is missing then return with the error.
        if (empty($request->get('id'))) {
            $response = $this->default_function->push_error($response, 'Undefined User');
        }

        ## return error if we've it.
        if (!empty($response)) {
            return $response;
        }

        /** @var User $User */
        try {
            $response = $this->user_repository->find($request->get('id'));
        } catch (\Exception $exception) {
            $response = $this->default_function->push_error($response, $exception->getMessage());
        }

        return $response;
    }

    /**
     * Delete user.
     *
     * @param ParameterBag $request
     *
     * @return string
     */
    public function deleteUser(ParameterBag $request)
    {
        $User = $this->pre__userCheck($request);

        if ($User instanceof User) {
            ## if we found the user

            ## get the custom field reference, which we've to lock
            try {
                $User->setIsDeleted(!$User->getIsDeleted());
                $this->entity_manager->persist($User);
                $this->entity_manager->flush();
                $backResponse = 'OK';
            } catch (\Exception $exception) {
                $backResponse = $exception->getMessage();
            }

        } else {
            ## if we didn't find the user then return the error.
            $backResponse = 'Invalid User Details';
        }

        return $backResponse;
    }


    ## get the user
    public function get__user(int $reference, ParameterBag $args = null)
    {
        $response = 'Invalid user reference';
        if (!empty($reference)) {
            try {
                ## find on result who is matching with this criteria
                if (!empty($args) && $args->get('findOne', false) && !empty($args->get('criteria'))) {
                    $response = $this->user_repository->findOneBy($args->get('criteria'));
                } else {
                    ## find only on record only by id
                    $response = $this->user_repository->find($reference);
                }

            } catch (\Exception $exception) {
                $response = $exception->getMessage();
            }
        }

        return $response;
    }


    /**
     *  ======================= Common Methods.
     */


    /**
     * @param array $idsOfUser
     * PURPOSE: Get the list of user, from the user id.
     *
     * @return array
     */
    public function getBulkUserListFromUserIds(array $idsOfUser = [])
    {
        $returnArr = [];
        foreach ($idsOfUser as $key => $value) {
            $bag = new ParameterBag();
            $bag->set('id', $value);
            $user = $this->pre__userCheck($bag);
            if ($user instanceof User) {
                array_push($returnArr, $user);
            }
        }

        return $returnArr;
    }



    /**
     * ====================== Forgot password
     * */

    ## validate forgot password form
    public function validate__forgot__password(ParameterBag $request)
    {
        $response = [];

        ## email is valid
        if (empty($request->get('--e')) || !filter_var($request->get('--e'), FILTER_VALIDATE_EMAIL)) {
            $response = $this->default_function->push_error($response, 'Please type you email');
        }

        ## csrf token
        if (empty($request->get('_token_csrf_'))) {
            $response = $this->default_function->push_error($response, 'Invalid CSRF token');
        }

        return $response;
    }

    ## request for forgot password
    public function request___forgot_password(ParameterBag $request)
    {
        $response = [];

        ## validation response.
        if ($validtion_response = $this->validate__forgot__password($request)) {
            return $validtion_response;
        }

        $token = new CsrfToken(getenv('APP_SECRET'), $request->get('_token_csrf_'));

        ## validate CSRF token.
        if (!$this->csrf_token_manager->isTokenValid($token)) {
            $response = $this->default_function->push_error($response, 'Invalid CSRF token');
        }

        ## if response is not empty
        if (!empty($response)) {
            return $response;
        }

        ## validate email existence in database
        $User = $this->user_repository->findOneBy(['email' => $request->get('--e')]);
        if (!$User instanceof User) {
            $response = $this->default_function->push_error($response, 'User not exits');
        }

        ## if response is not empty
        if (!empty($response)) {
            return $response;
        }

        $bag = new ParameterBag();
        ## generate token and save to variable.
        $bag->set('tk__', $this->custom_token_manager->generate_token($User->getEmail()));
        $bag->set('gOc', 'forgot_password');
        $bag->set('user', $User);

        ## save the token  in the database
        $token_response = $this->custom_token_manager->save__token($bag);

        ## getting the save token if record is saved successfully.
        if (is_string($token_response) && substr($token_response, 0, 2) == 'OK') {
            ## send email with the token
            ## send notification to user, on registration.
            $temp_ar = [
                'event_name' => 'FORGOT_PASSWORD',
                'user' => $User,
                'additionalInfo' => [
                    '{{ resetPasswordLink }}' => $this->url_generator->generate(
                        'security__update_my_pass_by_resetPass',
                        ['token__string' => substr($token_response, 2)]
                    ),
                ],
            ];
            $response = $this->trigger_notifications->triggerNotifications($temp_ar);
            $response = $response ? 'OK' : $response;
        } else {
            $response = $token_response;
        }

        return $response;
    }

    ## do some actions before udpate new password when user request to reset password
    public function updateUserPassword__pre__action(ParameterBag $request)
    {
        ## validate token..
        $response = [];

        ## csrf token
        if (empty($request->get('_token_csrf_'))) {
            $response = $this->default_function->push_error($response, 'Invalid CSRF token');
        } else {
            $token = new CsrfToken(getenv('APP_SECRET'), $request->get('_token_csrf_'));

            ## validate CSRF token.
            if (!$this->csrf_token_manager->isTokenValid($token)) {
                $response = $this->default_function->push_error($response, 'Invalid CSRF token');
            }
        }

        ## if we've errors.
        if (!empty($response)) {
            return $response;
        }

        ## decrypt token.
        $email = $this->custom_token_manager->decrypt__token($request->get('token_string'));
        try {
            $User = $this->user_repository->findOneBy(['email' => $email]);
        } catch (\Exception $exception) {
            $response = $exception->getMessage();
        }


        ## if we've errors.
        if (!empty($response)) {
            return $response;
        }

        ## don''t ask current password.k
        $request->set('ask___cp', '___sm');
        ## update the password
        $response = $this->updateUserPassword($request, $User);

        if ($response == 'OK') {
            ## update the one time token option.
            $response = $this->custom_token_manager->expire__OneTimeUsed__token__($request->get('token_string'));
        }

        return $response;
    }


    /**
     * ================= Teachers
     * */
    public function getTeachers()
    {
        return $this->user_repository->findBy(['category' => TeacherController::userCategory]);
    }

    /**
     * ================= Students
     * */
    public function getStudents()
    {
        return $this->user_repository->findBy(['category' => StudentController::userCategory]);
    }

    /**
     * ================= Admins
     * */
    public function getAdmins()
    {
        return $this->user_repository->findBy(['category' => AdminController::userCategory]);
    }

}
