<?php

namespace App\Security;

use App\Entity\Tokens;
use App\Entity\User;
use App\Service\CustomTokenManager;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Exception\CustomUserMessageAuthenticationException;
use Symfony\Component\Security\Core\Encoder\UserPasswordEncoderInterface;
use Symfony\Component\Security\Core\Exception\InvalidCsrfTokenException;
use Symfony\Component\Security\Core\Security;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Security\Core\User\UserProviderInterface;
use Symfony\Component\Security\Csrf\CsrfToken;
use Symfony\Component\Security\Csrf\CsrfTokenManagerInterface;
use Symfony\Component\Security\Guard\Authenticator\AbstractFormLoginAuthenticator;
use Symfony\Component\Security\Http\Util\TargetPathTrait;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Response;

class ApiLoginAuthAuthenticator extends AbstractFormLoginAuthenticator
{
    use TargetPathTrait;

    private $entityManager;
    private $urlGenerator;
    /**
     * @var ApiTokenGenerator
     */
    private $apiTokenGenerator;
    /**
     * @var CustomTokenManager
     */
    private $customTokenManager;

    public function __construct(
        EntityManagerInterface $entityManager,
        UrlGeneratorInterface $urlGenerator,
        ApiTokenGenerator $apiTokenGenerator,
        CustomTokenManager $customTokenManager
    ) {
        $this->entityManager = $entityManager;
        $this->urlGenerator = $urlGenerator;
        $this->apiTokenGenerator = $apiTokenGenerator;
        $this->customTokenManager = $customTokenManager;
    }

    public function supports(Request $request)
    {
        ## get the header from request
        $apiToken = $request->headers->has('X-AUTH-TOKEN');

        if (!$apiToken) {
            ## if api token not found, in the header and route is login route then logged in the user, by hint the route.
            if ($request->attributes->get('_route') === 'api_login_route') {
                ## return the generated Login Token after or error if username and password are not matched.
                $generatedToken = $this->apiTokenGenerator->generateToken($request);
                $request->headers->set('X-AUTH-TOKEN', $generatedToken);
                $response = true;
            } else {
                $response = false;
            }
        } else {
            ## yes, API token has found now go forward.
            $response = true;
        }

        return $response;
    }

    public function getCredentials(Request $request)
    {
        return [
            'token' => $request->headers->get('X-AUTH-TOKEN'),
        ];
    }

    public function getUser($credentials, UserProviderInterface $userProvider)
    {
        $apiToken = $credentials['token'];
        if (null === $apiToken) {
            return false;
        }

        // if a User object, checkCredentials() is called
        $userObject = $this->entityManager->getRepository(Tokens::class)
            ->findOneBy(['token_name' => $apiToken]);
        try {
            $tokenValidationTest = $this->customTokenManager->is__token__valid($apiToken);
            $tokenValidationTest = !$tokenValidationTest ?? null;
        } catch (\Exception $e) {
            $tokenValidationTest = null;
        }
        $userObject = !$tokenValidationTest ? $userObject : null;
        if ($userObject instanceof Tokens) {
            $userObject = $this->entityManager->getRepository(User::class)
                ->find($userObject->getUser()->getId());
        }

        return $userObject;
    }

    public function checkCredentials($credentials, UserInterface $user)
    {
        if ($user instanceof User) {
            return true;
        } else {
            return null;
        }
    }

    public function onAuthenticationSuccess(Request $request, TokenInterface $token, $providerKey)
    {
        return null;
    }

    /**
     * Called when authentication is needed, but it's not sent
     * @param Request $request
     * @param AuthenticationException|null $authException
     * @return JsonResponse
     */
    public function start(Request $request, AuthenticationException $authException = null)
    {
        $data = [
            // you might translate this message
            'message' => 'Authentication Required',
        ];

        return new JsonResponse($data, Response::HTTP_UNAUTHORIZED);
    }

    public function supportsRememberMe()
    {
        return false;
    }

    protected function getLoginUrl()
    {
        return $this->urlGenerator->generate('user_login');
    }
}
