<?php

namespace App\Controller;

use App\Entity\Assesment;
use App\Entity\ClassSessionEnrolment;
use App\Entity\Guardian;
use App\Entity\User;
use App\Repository\ClassesRepository;
use App\Repository\ClassSessionEnrolmentRepository;
use App\Repository\GuardianRepository;
use App\Repository\InvoicesRepository;
use App\Repository\UserRepository;
use App\Service\AppSettings;
use App\Service\Constants;
use App\Service\DefaultFunction;
use App\Service\FileManagment;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Mapping\Driver\DatabaseDriver;
use Knp\Bundle\SnappyBundle\Snappy\Response\PdfResponse;
use Knp\Snappy\Pdf;
use Nzo\UrlEncryptorBundle\Annotations\ParamDecryptor;
use Nzo\UrlEncryptorBundle\UrlEncryptor\UrlEncryptor;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\ParameterBag;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;


/**
 * @IsGranted("ROLE_USER")
 *
 * */
class StudentController extends AbstractController
{

    const userCategory = 's';
    /**
     * @var UserRepository
     */
    private $user_repository;
    /**
     * @var ClassSessionEnrolmentRepository
     */
    private $class_session_enrolment_repository;
    /**
     * @var UrlEncryptor
     */
    private $url_encryptor;
    /**
     * @var Guardian
     */
    private $guardian;
    /**
     * @var GuardianRepository
     */
    private $guardian_repository;
    /**
     * @var ClassesRepository
     */
    private $classes_repository;
    /**
     * @var Pdf
     */
    private $pdf;
    /**
     * @var AppSettings
     */
    private $appSettings;
    /**
     * @var EntityManagerInterface
     */
    private $entityManager;

    public function __construct(
        ClassesRepository $classes_repository,
        UserRepository $user_repository,
        ClassSessionEnrolmentRepository $class_session_enrolment_repository,
        UrlEncryptor $url_encryptor,
        GuardianRepository $guardian_repository,
        Pdf $pdf,
        AppSettings $appSettings,
        EntityManagerInterface $entityManager
    ) {
        $this->user_repository = $user_repository;
        $this->class_session_enrolment_repository = $class_session_enrolment_repository;
        $this->url_encryptor = $url_encryptor;
        $this->guardian_repository = $guardian_repository;
        $this->classes_repository = $classes_repository;
        $this->pdf = $pdf;
        $this->appSettings = $appSettings;
        $this->entityManager = $entityManager;
    }


    /**
     * @Route("/{_locale}/students/manage-student/", name="student_list", defaults={"loacle"="en"})
     *
     * @param Request $request
     *
     * @return Response
     */
    public function index(Request $request)
    {

        $Bag = new ParameterBag();
        $Bag->set('category', self::userCategory);
        $Bag->set('classSessionEnrolment', $this->url_encryptor->decrypt($request->get('csE')));
        $Bag->set('classSession', $this->url_encryptor->decrypt($request->get('cs')));
        $Bag->set('orderBy', $request->get('oB', 'id'));
        $Bag->set('order', $request->get('o', 'asc'));
        $Bag->set('studentName', $request->get('studentName'));
        $Bag->set('guardianRef', $request->get('guardian'));
        $Bag->set('activeAndInActive', $request->get('status'));
        $Bag->set('searchByGender', $request->get('gender'));
        $Bag->set('userAdmissionDateRange', $request->get('dateRange'));


        ## render student template.
        //return $this->render( 'student/index.html.twig', [
        return $this->render(
            'student/index.html.twig',
            [
                ## use in filters
                'students' => $this->user_repository->findBy(['category' => self::userCategory]),
                ## use in table view
                'UsersLIst' => $this->user_repository->studentList($Bag, $this->getUser()),
                ## Class Session enrolment
                'classSession' => $this->classes_repository->getAllClasses($this->getUser()),
                ## Guardian name
                'Guardian' => $this->guardian_repository->findAll(),
            ]
        );
    }

    /**
     * @Route("/{_locale}/api/students/", name="student_list_api", defaults={"loacle"="en"})
     * @param Request $request
     *
     * @return JsonResponse
     */
    public function student_list_api(Request $request)
    {
        ## get all students.
        $studentList = $this->user_repository->allUsers(self::userCategory, $request->query->get('query'), true);

        return new JsonResponse(
            $studentList, Response::HTTP_OK, [
                'content-type' => 'application/json',
            ]
        );

    }

    /**
     * @Route("{_locale}/student/{userID}" , name="single_student_user", requirements={"userID":"\d+"})
     * @param int|null $userID
     *
     * @return Response
     */
    public function singleStudent(?int $userID = 0)
    {

        ## if user id is valid
        if ($userID) {
            ## get student.
            $studentList = $this->user_repository->find($userID);
        } else {
            return $this->redirectToRoute('student_list');
        }

        ## render student template.
        //return $this->render( 'student/index.html.twig', [
        return $this->render(
            'student/index.html.twig',
            [
                'list' => $studentList,
            ]
        );
    }

    /**
     * @Route("{_locale}/student/report/{userID}" , name="studentDetailReport")
     * @ParamDecryptor(params={"userID"})
     * Reporting
     *
     * PURPOSE: a student full details, in which class he/she enroll, her attendance status, her fee status, her exam
     * details
     *
     * @param         $userID
     * @param Request $request
     *
     * @return PdfResponse|Response
     */
    public function studentDetailReport($userID, Request $request)
    {


        /** @var User $studentDetails */
        $studentDetails = $this->user_repository->getStudentDetailReport($userID);

        if ($request->query->get('print') == 1) {

            $html_fe = $this->renderView(
                'student/pdfReporting/studentDetailReport.html.twig',
                ['student' => $studentDetails]
            );

            //die($html_fe);

            return new Response(
                $this->pdf->getOutputFromHtml(
                    $html_fe,
                    [
                        'encoding' => 'utf-8',
                    ]
                ), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$studentDetails->getFullName().'_studentDetail.pdf"',
                ]
            );

        }


        ## render student report.
        //return $this->render( 'student/index.html.twig', [
        return $this->render(
            'student/studentDetailReport.html.twig',
            [
                'student' => $studentDetails,
            ]
        );
    }

    /**
     * @Route("{_locale}/student/payment-report/" , name="student_payment_report")
     *
     * PURPOSE: how much student has paid during a date
     *
     * @param Request $request
     * @param InvoicesRepository $invoicesRepository
     * @param DefaultFunction $defaultFunction
     *
     * @return Response
     */
    ## student payment report
    public function studentPaymentReport(
        Request $request,
        InvoicesRepository $invoicesRepository,
        DefaultFunction $defaultFunction
    ) {
        $bag = new ParameterBag();

        ## user id
        $userID = $this->url_encryptor->decrypt($request->query->get('userID'));

        $bag->set('DueDateRange', $request->query->get('date'));
        $bag->set('user', $userID);
        ## search invoice for the student only
        $bag->set('userCategory', 's');

        $paidInvoicesByUser = null;
        if (count($request->query)) {
            ## search records
            $paidInvoicesByUser = $invoicesRepository->search__inv($bag);
        }


        $sendingData = [
            'paidInvoiceByUser' => $paidInvoicesByUser,
            'date' => $bag->get('DueDateRange'),
        ];

        ## print pdf file
        if ($request->query->get('printResult') == 1) {

            ## sending extra data to the print.
            $templateName = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('StudentPaymentReport')
            );
            if (empty($templateName)) {
                $templateName = 'pdfPrints/student.root/Student.Payment.Report/default.html.twig';
            }

            $sendingData['student'] = $this->user_repository->find(
                $this->url_encryptor->decrypt($request->query->get('userID'))
            );
            $html_fe = $this->renderView($templateName, $sendingData);


            $filename = 'Student Payment Report';

            return new Response(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );
        }

        $sendingData['studentList'] = $this->user_repository->findBy(['category' => self::userCategory]);

        return $this->render('student/reporting/studentPaymentReport.html.twig', $sendingData);

    }

    /*
 * ================= Time Table
 * */

    /**
     * @Route("{_locale}/student/time-table" , name="student_time_table")
     *
     * @return PdfResponse|Response
     */
    public function studentTimeTable()
    {
        return $this->render('student/timeTable/timetable.html.twig', []);
    }

    /*
     * ================= Assessments
     * */

    /**
     * @Route("{_locale}/student/assessment/", name="student_assessment")
     * @param Request $request
     * @param FileManagment $fileManagement
     * @return Response
     */
    public function student_assessment(Request $request, FileManagment $fileManagement)
    {
        /**@var User $user */
        $user = $this->getUser();

        if ($user->getCategory() === 'g') {
            /** @var Guardian $user */
            $users = $user->getStudentReference()->toArray();
        } else {
            $users = [$user];
        }

        ## Find Classes in which students are enrolled
        $classes = [];
        foreach ($users as $user) {
            $classEnrolments = $user->getClassSessionEnrolments()->toArray();
            /** @var ClassSessionEnrolment $classEnrolment */
            foreach ($classEnrolments as $classEnrolment) {
                array_push($classes, $classEnrolment->getClassSession()->getId());
            }
        }
        $classes = array_unique($classes);
        $assessment = [];
        foreach ($classes as $class) {
            ## find assessment from the assessment table.
            $assessments = $this->entityManager->getRepository(Assesment::class)->findBy(
                ['assesment_class' => $class]
            );

            /** @var Assesment $asses */
            foreach ($assessments as $asses) {
                $tmp_ar['assessment'] = $asses;
                array_push($assessment, $asses);
            }
        }

        return $this->render(
            'student/assessment/studentAssessment.html.twig',
            [
                'assessments' => $assessment,
            ]
        );
    }

}
