<?php


namespace App\Service\Guardian;


use App\Entity\Guardian;
use App\Entity\User;
use App\Repository\GuardianRepository;
use App\Service\DefaultFunction;
use App\Service\UserService;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\ParameterBag;
use Symfony\Component\Security\Core\Encoder\UserPasswordEncoder;
use Symfony\Component\Security\Core\Encoder\UserPasswordEncoderInterface;

class GuardianService
{

    /**
     * @var DefaultFunction
     */
    private $default_function;
    /**
     * @var GuardianRepository
     */
    private $guardian_repository;
    /**
     * @var UserPasswordEncoder
     */
    private $user_password_encoder;
    /**
     * @var EntityManagerInterface
     */
    private $entity_manager;
    /**
     * @var UserService
     */
    private $user_service;

    /**
     * GuardianService constructor.
     *
     * @param DefaultFunction $default_function
     * @param GuardianRepository $guardian_repository
     * @param UserPasswordEncoderInterface $user_password_encoder
     * @param EntityManagerInterface $entity_manager
     */
    public function __construct(
        DefaultFunction $default_function,
        GuardianRepository $guardian_repository,
        UserPasswordEncoderInterface $user_password_encoder,
        EntityManagerInterface $entity_manager
    ) {
        $this->default_function = $default_function;
        $this->guardian_repository = $guardian_repository;
        $this->user_password_encoder = $user_password_encoder;
        $this->entity_manager = $entity_manager;
        //$this->user_service          = $user_service;
    }


    /**
     * @param ParameterBag $bag
     * PURPOSE: Validate the guardian registration.
     *
     * @return array|mixed
     */
    public function validateGuardianRegistration(ParameterBag $bag)
    {
        $response = [];

        ## first name exist
        if (empty($bag->get('GfName'))) {
            $response = $this->default_function->push_error($response, 'Guardian first name is missing');
        }

        ## Last name exist
        if (empty($bag->get('GlName'))) {
            $response = $this->default_function->push_error($response, 'Guardian last name is missing');
        }

        ## email
        if (empty($bag->get('GEmail')) || !filter_var($bag->get('GEmail'), FILTER_VALIDATE_EMAIL)) {
            $response = $this->default_function->push_error($response, 'Guardian email name is missing');
        }


        ## phone
        if (empty($bag->get('GPhone'))) {
            $response = $this->default_function->push_error($response, 'Guardian phone number is missing');
        }

        ## Guardian relation
        if (empty($bag->get('GR'))) {
            $response = $this->default_function->push_error($response, 'Guardian relation is require');
        }


        return $response;
    }

    /**
     * @param ParameterBag $bag
     * PURPOSE: registering guardian or update the guardian.
     *
     * @return string
     */
    public function registerGuardian(ParameterBag $bag)
    {
        ## check validation
        if (!empty($validationResponse = $this->validateGuardianRegistration($bag))) {
            return $validationResponse;
        }

        $response = [];
        $Guardian = null;

        ## if guardian reference existence
        if (!empty($bag->get('GRe'))) {
            $Guardian = $this->getGuardian($bag->get('GRe'));
        }

        ## if Guardian is still not found
        if (!$Guardian instanceof Guardian) {
            $Guardian = new Guardian();
            ## make is active when we're adding a new guardian
            $Guardian->setIsActive(true);
        }


        ## get the student object
        if (!empty($bag->get('GUref'))) {
            $Student = $this->user_service->get__user((int)$bag->get('GUref'));
            if (!$Student instanceof User) {
                $response = $this->default_function->push_error($response, 'Student reference not found');
            }
            $Guardian->addStudentReference($Student);
        } elseif (!empty($bag->get('studentObject'))) {
            $Guardian->addStudentReference($bag->get('studentObject'));
        }


        ## duplication check.
        ## find the user of same email or phone, if exits then throw a message user already exists.
        if (!empty(
        $this->guardian_repository->findNotOneBy(
            'guardian_email',
            $bag->get('GEmail'),
            empty($Guardian->getId()) ? 0 : $Guardian->getId()
        )
        )) {
            $response = $this->default_function->push_error(
                $response,
                sprintf('%s This email is already register with other guardian', $bag->get('GEmail'))
            );
        }
        if (!empty(
        $this->guardian_repository->findNotOneBy(
            'phone',
            $bag->get('GPhone'),
            empty($Guardian->getId()) ? 0 : $Guardian->getId()
        )
        )) {
            $response = $this->default_function->push_error(
                $response,
                sprintf('%s This phone is already register with other guardian', $bag->get('GPhone'))
            );
        }

        ## return if error is not exits;
        if (!empty($response)) {
            return $response;
        }

        ## encrypt the password.
        try {
            $Guardian->setFname($bag->get('GfName'));
            $Guardian->setLname($bag->get('GlName'));
            $Guardian->setGuardianEmail($bag->get('GEmail'));
            $Guardian->setGuardianRelation($bag->get('GR'));
            $Guardian->setRoles(['ROLE_GUARDIAN']);
            $Guardian->setPhone(rtrim($bag->get('GPhone')));
            $Guardian->setGender($bag->get('gender'));
            $Guardian->setPassword('Testing');
            $Guardian->setIsActive($bag->get('GActive', true));

            ## save the guardian in the database.
            $this->entity_manager->persist($Guardian);
            $this->entity_manager->flush();

            ## update the password convert it into hash
            $Guardian->setPassword(
                $this->user_password_encoder->encodePassword($Guardian, $bag->get('secret', 'Testing'))
            );
            $this->entity_manager->persist($Guardian);
            $this->entity_manager->flush();

            $response = 'OK';
        } catch (\Exception $exception) {
            $response = $exception->getMessage();
        }

        return $response;
    }

    /**
     * @param int $Ref
     * PURPOSE: getting the guardian from database
     *
     * @param bool $getGuardianUserByEmail
     * @param bool $getGuardianByPhone
     *
     *
     * @return Guardian|string|null
     */
    public function getGuardian($Ref, $getGuardianUserByEmail = false, $getGuardianByPhone = false)
    {
        $response = null;
        try {

            if (!empty($getGuardianByPhone)) {
                ## get the guardian by phone.
                $response = $this->guardian_repository->findOneBy(['phone' => rtrim($Ref)]);
            } else {
                if (!empty($getGuardianUserByEmail)) {
                    ## get the guardian by email.
                    $response = $this->guardian_repository->findOneBy(['guardian_email' => $Ref]);
                } else {
                    ## get the guardian by id
                    $response = $this->guardian_repository->find($Ref);
                }
            }
        } catch (\Exception $exception) {
            $response = $exception->getMessage();
        }

        return $response;
    }

    /**
     * @param ParameterBag $request
     *
     * @param ParameterBag|null $requestUploadFile
     * @param UserService|null $userService
     *
     * @return array
     * PURPOSE: update guardian profile
     */
    public function updateProfile($request, ParameterBag $requestUploadFile = null, UserService $userService = null)
    {
        $response = [];

        ## get the user
        $User = $this->getGuardian($request->get('id'));

        ## validation.
        if (empty($request->get('fName'))) {
            $response = $this->default_function->push_error($response, 'First name is missing');
        }

        ## validation.
        if (empty($request->get('lname'))) {
            $response = $this->default_function->push_error($response, 'Last name is missing');
        }

        ## validation.
        if (empty($request->get('gender'))) {
            $response = $this->default_function->push_error($response, 'Please select Gender');
        }

        ## validate phone
        if (empty($request->get('phone'))) {
            $response = $this->default_function->push_error($response, 'Please type phone number');
        }

        ## email validation - check if email of this is user is already exits or not.
        if (!empty($request->get('email')) && (!filter_var($request->get('email'), FILTER_VALIDATE_EMAIL))) {
            ##push errors in the array
            $response = $this->default_function->push_error(
                $response,
                $request->get('email').' ,  Invalid email address'
            );
        }

        ## email already existence check
        $emailCheck = $this->guardian_repository->getGuardianWhereGuardianNotByID(
            $User->getId(),
            $request->get('email')
        );
        if (!empty($emailCheck)) {
            $response = $this->default_function->push_error(
                $response,
                $request->get('email').' , is already register with the other user'
            );
        }
        ## phone existence check
        if (!empty(
        $this->guardian_repository->getGuardianWhereGuardianNotByID(
            $User->getId(),
            null,
            $request->get('phone')
        )
        )) {
            $response = $this->default_function->push_error(
                $response,
                $request->get('phone').' phone, is already register with the other user'
            );
        }

        ## return error if we've it.
        if (!empty($response)) {
            return $response;
        }


        ## set user f and l name, email and gender.
        $User->setFName($request->get('fName'));
        $User->setLName($request->get('lname'));
        $User->setEmail($request->get('email'));
        $User->setPhone($request->get('phone'));
        $User->setGender($request->get('gender'));

        // if file array is not empty
        if (!empty($requestUploadFile->get('profile_avatar'))) {
            $response_ = $userService->uploadUserProfile(
                $requestUploadFile->get('profile_avatar'),
                $request->get('id'),
                'g'
            );
            if ($response_ <> 'OK' || (is_array(
                        $response_
                    ) && !empty($response_['response']) && $response_['response'] <> 'OK')) {
                $response = $this->default_function->push_error($response, $response_['response']);
            }
        }

        ## return error if we've it.
        if (!empty($response)) {
            return $response;
        }

        try {
            ## save user in a database.
            $this->entity_manager->persist($User);
            $this->entity_manager->flush();

        } catch (\Exception $exception) {
            $response = $this->default_function->push_error($response, $exception->getMessage());
        }

        ## if we've no errors then return
        if (empty($response)) {
            $response = 'OK'.$User->getId();
        }

        return $response;

    }

    /**
     * @param array $studentList
     * PURPOSE: get guardians of students
     *
     * @return array
     */
    public function getGuardiansFromStudentRef(array $studentList)
    {
        $returnAr = [];
        foreach ($studentList as $key => $value) {
            if ($value instanceof User) {
                $returnAr = array_merge($returnAr, $value->getGuardians()->toArray());
            }
        }

        return $returnAr;
    }


    public static function getAttachedStudentsIdWithGuardian($guardian)
    {
        $tempAr = [];
        if ($guardian instanceof Guardian) {
            $attachedStudents = $guardian->getStudentReference()->toArray();
            foreach ($attachedStudents as $key => $value) {
                array_push($tempAr, $value->getId());
            }
        }

        return $tempAr;
    }
}
