<?php

namespace App\Repository;

use App\Entity\User;
use App\Service\DefaultFunction;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\DBAL\DBALException;
use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\NonUniqueResultException;
use Symfony\Bridge\Doctrine\RegistryInterface;
use Symfony\Component\HttpFoundation\ParameterBag;

/**
 * @method User|null find($id, $lockMode = null, $lockVersion = null)
 * @method User|null findOneBy(array $criteria, array $orderBy = null)
 * @method User[]    findAll()
 * @method User[]    findBy(array $criteria, array $orderBy = null, $limit = null, $offset = null)
 *
 * TableDOC:
 * Columns Description:
 * Category: this column will description which type of use it is? s for student, t for teacher and a for administrator
 */
class UserRepository extends ServiceEntityRepository
{

    /**
     * @var RegistryInterface
     */
    private $registry;
    /**
     * @var DefaultFunction
     */
    private $default_function;
    /**
     * @var AllocateTeacherToClassAndSubjectsRepository
     */
    private $allocateTeacherToClassAndSubjectsRepository;

    public function __construct(
        RegistryInterface $registry,
        DefaultFunction $default_function,
        AllocateTeacherToClassAndSubjectsRepository $allocateTeacherToClassAndSubjectsRepository
    ) {
        parent::__construct($registry, User::class);
        $this->registry = $registry;
        $this->default_function = $default_function;
        $this->allocateTeacherToClassAndSubjectsRepository = $allocateTeacherToClassAndSubjectsRepository;
    }


    /**
     * @deprecated 01-Nov-2019
     * */
    public function search($term)
    {
        $this->createQueryBuilder('u')
            ->join('u.sAnswerRegardingCFs', 'Stu')
            ->addSelect('Stu')
            ->andWhere('u.fName like :term')
            ->orWhere('u.lName like :term')
            ->orWhere('u.email like :term')
            ->orWhere('Stu.answers like :term')
            ->setParameter('term', '%'.$term.'%');
    }

    /**
     * @deprecated 01-Nov-2019
     * */
    public function CustomFindAll()
    {
        $term = 'male';


        return $this->createQueryBuilder('u')## join with the student answer
        ->innerJoin('u.sAnswersRegardingCFs', 'studentAnswerTable')
            ->addSelect('studentAnswerTable')## join with the custom fields.
            ->innerJoin('studentAnswerTable.CustomFields', 'custom_fields')
            ->addSelect('custom_fields')## search codes.
            ->andWhere('u.fName like :term')
            ->orWhere('u.lName like :term')
            ->orWhere('studentAnswerTable.answers like :term')
            ->orWhere('custom_fields.fLabel like :term')
            ->setParameter('term', '%'.$term.'%')
            ->getQuery()
            ->getResult();

    }

    /**
     * @deprecated 01-Nov-2019
     * */
    public function getUserAdditionalData($userId)
    {
        return $this->createQueryBuilder('user')
            ->leftJoin('user.sAnswersRegardingCFs', 'sanswers_regarding_cfs')
            ->andWhere('user.id', $userId)
            ->getQuery()
            ->getResult();
    }

    /**
     * @deprecated 01-Nov-2019
     * */
    public function advanceSearch(ParameterBag $bag)
    {
        $queryBuilder = $this->createQueryBuilder('user');

    }


    /**
     * @param string|null $userCategory
     *
     * @param string|null $searchTerm
     *
     *
     * @return User|null
     *
     *  Purpose: This method is used to fetch all users or user according to its category
     */
    public function allUsers(?string $userCategory = null, ?string $searchTerm = null, ?bool $arrayList = false)
    {

        ## create query builder
        $queryBuilder = $this->createQueryBuilder('u');

        ## to access the student answers related to the fetched users
        $queryBuilder->leftjoin('u.sAnswersRegardingCFs', 'studentAnswers')->addSelect('studentAnswers');

        ## I used only join keyword (inner join) but its not work through doctrine
        ## but when i execute doctrine generated on mysql database with only join keyword, its works.
        $queryBuilder->leftjoin('studentAnswers.CustomFields', 'customFields')->addSelect('customFields');

        if ($userCategory) {
            ## if user category is not empty
            $queryBuilder->andWhere('u.category = :userCategory')->setParameter('userCategory', $userCategory);
        }

        ## omit those users which are not-deleted.
        $queryBuilder->andWhere('u.is_deleted IS NULL')
            ->orWhere('u.is_deleted = :is_deleted')
            ->setParameter('is_deleted', 0);

        if ($searchTerm) {
            ## search term
            $queryBuilder->andWhere('( u.fName like :searchTerm')
                ->orWhere('u.lName like :searchTerm')
                ->orWhere('studentAnswers.answers like :searchTerm')
                ->orWhere('customFields.fLabel like :searchTerm )')
                ->setParameter('searchTerm', '%'.$searchTerm.'%');
        }

        ## make query & get the all results.
        if ($arrayList) {
            $response = $queryBuilder->getQuery()->getArrayResult();
        } else {
            $response = $queryBuilder->getQuery()->getResult();
        }

        return $response;
    }

    ## get all users except the current logged in user.
    public function get__all_users_except_current_loggedin_user(int $exceptUser)
    {
        return $this->createQueryBuilder('user')
            ->andWhere('user.id != :userId')
            ->setParameter('userId', $exceptUser)
            ->orderBy('user.fName', 'asc')
            ->getQuery()
            ->getResult();
    }

    /**
     * get un deleted users.
     * */
    public function getUnDeletedUsers()
    {
        return $this->createQueryBuilder('user')
            ->andWhere('user.is_deleted = :is_deleted')
            ->setParameter('is_deleted', 0)
            ->getQuery()
            ->getResult();
    }

    ## associated user with the email.
    public function getAssociatedUserWithEmail($email, $user_id)
    {
        $queryBuilder = $this->createQueryBuilder('user');

        ## if email is given
        if (!empty($email)) {
            $queryBuilder->andWhere('user.email = :email')
                ->setParameter('email', $email);
        }

        ## user id is given
        if (!empty($user_id)) {
            $queryBuilder->andWhere('user.id != :user_id')
                ->setParameter('user_id', $user_id);
        }

        return $queryBuilder->getQuery()->getResult();
    }
    ## find Not One By
    ## this method is used to find out there where "$value" of specific column is not equal to, of special user.
    public function findNotOneBy(string $key, $value, int $user_id)
    {
        return $this->createQueryBuilder('user')
            ->where(sprintf('user.%s= :value', $key))
            ->andWhere('user.id != :user_id')
            ->setParameter('value', $value)
            ->setParameter('user_id', $user_id)
            ->getQuery()
            ->getResult();

    }


    /**
     * ======================= STUDENT OPERATIONS.
     * */

    /**
     * Student List
     * PURPOSE: Use to show student list and and help in search
     *
     * @param ParameterBag $bag
     *
     * @param User $currentLoggedInUser
     * @return mixed
     */
    public function studentList(ParameterBag $bag, User $currentLoggedInUser = null)
    {

        $queryBuilder = $this->createQueryBuilder('user');

        ## get the guardians details of students
        $queryBuilder->leftJoin('user.guardians', 'guardians')->addSelect('guardians');

        ## get active class enrollment of students. - how can we get that ?
        ## We can get active class session enrollment on base of active session, if session is active means user is enrolled in this class.
        $queryBuilder->leftJoin('user.classSessionEnrolments', 'classSessionEnrolments')
            ->addSelect('classSessionEnrolments')
            ->leftJoin('classSessionEnrolments.class_session', 'classSession')
            ->addSelect('classSession')
            ->leftJoin('classSession.session', 'session')
            ->addSelect('session');

        ## Search by user's category
        if (!empty($bag->get('category'))) {
            $queryBuilder->andwhere('user.category = :cat')->setParameter('cat', $bag->get('category'));
        }

        ## Search classes based students.
        if (!empty($bag->get('classSessionEnrolment'))) {
            $queryBuilder->andwhere('classSessionEnrolments = :classEnrolment')
                ->setParameter('classEnrolment', $bag->get('classSessionEnrolment'));
        }

        ## Student name search
        if (!empty($bag->get('studentName'))) {
            $queryBuilder->andWhere('user.id = :userRef')->setParameter('userRef', $bag->get('studentName'));
        }

        ## Search student by guardian
        if (!empty($bag->get('guardianRef'))) {
            $queryBuilder->andwhere('guardians.id = :guardian')
                ->setParameter('guardian', $bag->get('guardianRef'));
        }

        ## active or inactive search
        if (!empty($bag->get('activeAndInActive'))) {
            $queryBuilder->andwhere('user.is_active = :activeStatus')
                ->setParameter('activeStatus', $this->default_function->parse__boolean($bag->get('activeAndInActive')));
        }

        ## Search by gender
        if (!empty($bag->get('searchByGender'))) {
            $queryBuilder->andwhere('user.gender = :genderSearch')
                ->setParameter('genderSearch', $bag->get('searchByGender'));
        }

        ## Date Range.
        ## Admission date search
        if (!empty($bag->get('userAdmissionDateRange'))) {

            ## Convert string into an array
            $studentAdmission = explode(' / ', $bag->get('userAdmissionDateRange'));

            ## if array is created and duration date is exits then search in database
            if (is_array($studentAdmission) && !empty($studentAdmission[0] && !empty($studentAdmission[1]))) {
                ## used in the duration
                $queryBuilder->andWhere('user.register_time BETWEEN :regFrom and :regTo');
                $queryBuilder->setParameter('regFrom', date('Y-m-d', strtotime(trim($studentAdmission[0]))).'%');
                $queryBuilder->setParameter('regTo', date('Y-m-d', strtotime(trim($studentAdmission[1]))).'%');
            }

        }

        ## Server Data against the User ROLE
        if ($currentLoggedInUser instanceof User && in_array('ROLE_TEACHER', $currentLoggedInUser->getRoles())) {
            ## if role is teacher then, server only those student who has enrolled in that class
            $allocatedClassesToTeacher = $this->allocateTeacherToClassAndSubjectsRepository->getUniqueClassAllocationByTeacherID(
                $currentLoggedInUser->getId()
            );
            if (is_array($allocatedClassesToTeacher) && !empty($allocatedClassesToTeacher)) {
                $queryBuilder->andWhere(
                    $queryBuilder->expr()->in('classSessionEnrolments.class_session', $allocatedClassesToTeacher)
                );
            }
        } else {
            if (!empty($bag->get('classSession'))) {
                ## Class Session search -- specific permission
                $queryBuilder->andWhere('classSessionEnrolments.class_session = :classSession')
                    ->setParameter('classSession', $bag->get('classSession'));
            }
        }


        ## Order of the result
        $queryBuilder->orderBy(
            empty($bag->get('orderBy')) ? 'user.id' : 'user.'.$bag->get('orderBy'),
            $bag->get('order')
        );

        return $queryBuilder->getQuery()->getResult();
    }


    /*
     * ================= ATTENDANCE ============
     * */

    /**
     * PURPOSE: Full report of student, where he/she enrolled, what is the exam report, attendance report and etc
     *
     * @param int $studentRef
     * @return mixed
     */
    public function getStudentDetailReport(int $studentRef)
    {
        try {
            $response = $this->createQueryBuilder('user')
                ->leftjoin('user.guardians', 'guardians')->addSelect('guardians')
                ->leftJoin('user.classSessionEnrolments', 'classSessionEnrolments')->addSelect('classSessionEnrolments')
                ->leftJoin('user.attendances', 'attendances')->addSelect('attendances')
                ->leftJoin('user.library_book_assigned_to', 'library_book_assigned_to')->addSelect(
                    'library_book_assigned_to'
                )
                ->leftJoin('user.invoices', 'invoices')->addSelect('invoices')
                ->leftJoin('user.examsClassesStudents', 'examsClassesStudents')->addSelect('examsClassesStudents')
                ->where('user.id = :userRef')->setParameter('userRef', $studentRef)
                ->getQuery()
                ->getOneOrNullResult();
        } catch (NonUniqueResultException $e) {
            $response = $e->getMessage();
        }

        return $response;

    }



    /*
     * =================================Teacher Specific operations
     *
     * */

    /**
     *
     * @param ParameterBag $bag
     * @return mixed
     */
    public function loadAttendanceOfTeacherAndStaff(ParameterBag $bag)
    {
        $connection = $this->getEntityManager()->getConnection();
        $in_date_sub_query = null;
        $parameters_to_add = [];

        ## if user want to filter their with respect to check_in_date
        try {
            $in__date = new \DateTime($bag->get('in_date'));
            $in__date = $in__date->format('Y-m-d').'%';

            ## sub query to filter out the date
            $in_date_sub_query = $connection->createQueryBuilder()
                ->select('*,id as attendance_ref')
                ->from('attendance')
                ->andWhere('attendance.datetime_in LIKE :in_date');
            ## add variable value in the array
            $parameters_to_add['in_date'] = $in__date;
        } catch (\Exception $exception) {
            return $exception->getMessage();
        }


        ## sub query to get the attendance status
        $attendance_status_sub_query = $connection->createQueryBuilder()
            ->select('*,id as attendance_status_id, name as attendance_status_name')
            ->from('attendance_status');

        ## get the user details which is attached in the class enrollment.
        $query____ = $connection->createQueryBuilder()
            ->select('*,user.id as userID')
            ->from('user', 'user')->andWhere('user.category = :userCatRef')->andWhere('user.is_deleted = 0')->andWhere(
                'user.is_active = 1'
            )->setParameter('userCatRef', $bag->get('userCatRef', 't'));

        ## to the get the attendance status.
        $query____->leftJoin(
            'attendance',
            sprintf('(%s)', $attendance_status_sub_query),
            'attendance_status',
            'attendance.attendance_status_id = attendance_status.id'
        );

        ## add sub query if found.
        if (!empty($in_date_sub_query)) {
            $query____->leftJoin(
                'user',
                sprintf('(%s)', $in_date_sub_query->getSQL()),
                'attendance',
                'user.id = attendance.user_id'
            );
        }

        ## set query parameters.
        foreach ($parameters_to_add as $key => $value) {
            $query____->setParameter($key, $value);
        }

        $s = $query____->execute();

        return $s->fetchAll();
    }


    /* Dashboard Report Query*/
    public function getLastSixteenDaysUserRegistration()
    {
        $sql = "
                    SELECT days.day, count(user.id) as count
            FROM
              (select curdate() as day
               union select curdate() - interval 1 day
               union select curdate() - interval 2 day
               union select curdate() - interval 3 day
               union select curdate() - interval 4 day
               union select curdate() - interval 5 day
               union select curdate() - interval 6 day
               union select curdate() - interval 7 day
               union select curdate() - interval 8 day
               union select curdate() - interval 9 day
               union select curdate() - interval 10 day
               union select curdate() - interval 11 day
               union select curdate() - interval 12 day
               union select curdate() - interval 13 day
               union select curdate() - interval 14 day
               union select curdate() - interval 15 day) days
              left join (select * from user where user.category='s') user
               on days.day = Date(user.register_time)
            group by
              days.day
              ";
        try {
            $stmt = $this->getEntityManager()->getConnection()->prepare($sql);
            $stmt->execute();
            return $stmt->fetchAll();
        } catch (DBALException $e) {
            return $e->getMessage();
        }
    }
}
