<?php

namespace App\Entity;

use App\Repository\InvoiceItemsRepository;
use App\Repository\InvoicesRepository;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;


/**
 * @ORM\Entity(repositoryClass="App\Repository\InvoicesRepository")
 */
class Invoices
{

    /**
     * @ORM\Id()
     * @ORM\GeneratedValue()
     * @ORM\Column(type="integer")
     */
    private $id;

    /**
     * @ORM\Column(type="text", nullable=true)
     */
    private $inv_system_notes;

    /**
     * @ORM\Column(type="text", nullable=true)
     */
    private $inv_user_notes;

    /**
     * @ORM\Column(type="datetime")
     */
    private $inv_due_date;

    /**
     * @ORM\Column(type="datetime", nullable=true)
     */
    private $inv_generated_datetime;

    /**
     * @var \DateTime $created_at
     *
     * @Gedmo\Timestampable(on="create")
     * @ORM\Column(type="datetime")
     */
    private $created_at;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\User", inversedBy="invoices")
     */
    private $inv_generated_by;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\User", inversedBy="invoices_generated_for")
     */
    private $inv_generated_for;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\PaymentMethods", inversedBy="invoices")
     * @ORM\JoinColumn(nullable=false)
     */
    private $payment_method;

    /**
     * @ORM\OneToMany(targetEntity="App\Entity\Transactions", mappedBy="invoice_reference")
     */
    private $transactions;

    /**
     * @ORM\OneToMany(targetEntity="App\Entity\InvoiceItems", mappedBy="invoice_reference")
     */
    private $invoiceItems;

    /**
     * @ORM\Column(type="string", length=255)
     */
    private $inv_reference;

    /**
     * @ORM\Column(type="string", length=60, nullable=true)
     */
    private $invoice_generated_for_unregister_username;

    /**
     * @ORM\OneToMany(targetEntity="App\Entity\ClassFeeSchedule", mappedBy="invoice_id")
     */
    private $classFeeSchedules;

    /**
     * @ORM\OneToMany(targetEntity="App\Entity\ClassSessionEnrolmentFeeSchedule", mappedBy="invoice")
     */
    private $classSessionEnrolmentFeeSchedules;

    ## invoice amount, which has not been paid.
    private $invoiceAmountWhichIsNotPaid = 0;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\ClassSessionEnrolment", inversedBy="invoices")
     * @ORM\JoinColumn(nullable=true)
     */
    private $class_session_enrolment;

    /**
     * @ORM\Column(type="boolean")
     */
    private $is_cancel = false;


    public function __construct()
    {
        $this->transactions = new ArrayCollection();
        $this->invoiceItems = new ArrayCollection();
        $this->classFeeSchedules = new ArrayCollection();
        $this->classSessionEnrolmentFeeSchedules = new ArrayCollection();
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getInvSystemNotes(): ?string
    {
        return $this->inv_system_notes;
    }

    public function setInvSystemNotes(?string $inv_system_notes): self
    {
        $this->inv_system_notes = $inv_system_notes;

        return $this;
    }

    public function getInvUserNotes(): ?string
    {
        return $this->inv_user_notes;
    }

    public function setInvUserNotes(?string $inv_user_notes): self
    {
        $this->inv_user_notes = $inv_user_notes;

        return $this;
    }

    public function getInvDueDate(): ?\DateTimeInterface
    {
        return $this->inv_due_date;
    }

    public function setInvDueDate(\DateTimeInterface $inv_due_date): self
    {
        $this->inv_due_date = $inv_due_date;

        return $this;
    }

    public function getInvGeneratedDatetime(): ?\DateTimeInterface
    {
        return $this->inv_generated_datetime;
    }

    public function setInvGeneratedDatetime(?\DateTimeInterface $inv_generated_datetime): self
    {
        $this->inv_generated_datetime = $inv_generated_datetime;

        return $this;
    }

    public function getCreatedAt(): ?\DateTimeInterface
    {
        return $this->created_at;
    }

    public function setCreatedAt(\DateTimeInterface $created_at): self
    {
        $this->created_at = $created_at;

        return $this;
    }

    public function getInvGeneratedBy(): ?User
    {
        return $this->inv_generated_by;
    }

    public function setInvGeneratedBy(?User $inv_generated_by): self
    {
        $this->inv_generated_by = $inv_generated_by;

        return $this;
    }

    public function getInvGeneratedFor(): ?User
    {
        return $this->inv_generated_for;
    }

    public function setInvGeneratedFor(?User $inv_generated_for): self
    {
        $this->inv_generated_for = $inv_generated_for;

        return $this;
    }

    public function getPaymentMethod(): ?PaymentMethods
    {
        return $this->payment_method;
    }

    public function setPaymentMethod(?PaymentMethods $payment_method): self
    {
        $this->payment_method = $payment_method;

        return $this;
    }

    /**
     * @return Collection|Transactions[]
     */
    public function getTransactions(): Collection
    {
        return $this->transactions;
    }

    public function addTransaction(Transactions $transaction): self
    {
        if (!$this->transactions->contains($transaction)) {
            $this->transactions[] = $transaction;
            $transaction->setInvoiceReference($this);
        }

        return $this;
    }

    public function removeTransaction(Transactions $transaction): self
    {
        if ($this->transactions->contains($transaction)) {
            $this->transactions->removeElement($transaction);
            // set the owning side to null (unless already changed)
            if ($transaction->getInvoiceReference() === $this) {
                $transaction->setInvoiceReference(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection|InvoiceItems[]
     */
    public function getInvoiceItems(): Collection
    {
        return $this->invoiceItems;
    }

    public function addInvoiceItem(InvoiceItems $invoiceItem): self
    {
        if (!$this->invoiceItems->contains($invoiceItem)) {
            $this->invoiceItems[] = $invoiceItem;
            $invoiceItem->setInvoiceReference($this);
        }

        return $this;
    }

    public function removeInvoiceItem(InvoiceItems $invoiceItem): self
    {
        if ($this->invoiceItems->contains($invoiceItem)) {
            $this->invoiceItems->removeElement($invoiceItem);
            // set the owning side to null (unless already changed)
            if ($invoiceItem->getInvoiceReference() === $this) {
                $invoiceItem->setInvoiceReference(null);
            }
        }

        return $this;
    }

    public function getInvReference(): ?string
    {
        return $this->inv_reference;
    }

    public function setInvReference(string $inv_reference): self
    {
        $this->inv_reference = $inv_reference;

        return $this;
    }

    public function getInvoiceGeneratedForUnregisterUsername(): ?string
    {
        return $this->invoice_generated_for_unregister_username;
    }

    public function setInvoiceGeneratedForUnregisterUsername(?string $invoice_generated_for_unregister_username): self
    {
        $this->invoice_generated_for_unregister_username = $invoice_generated_for_unregister_username;

        return $this;
    }

    /**
     * @return Collection|ClassFeeSchedule[]
     */
    public function getClassFeeSchedules(): Collection
    {
        return $this->classFeeSchedules;
    }

    public function addClassFeeSchedule(ClassFeeSchedule $classFeeSchedule): self
    {
        if (!$this->classFeeSchedules->contains($classFeeSchedule)) {
            $this->classFeeSchedules[] = $classFeeSchedule;
            $classFeeSchedule->setInvoiceId($this);
        }

        return $this;
    }

    public function removeClassFeeSchedule(ClassFeeSchedule $classFeeSchedule): self
    {
        if ($this->classFeeSchedules->contains($classFeeSchedule)) {
            $this->classFeeSchedules->removeElement($classFeeSchedule);
            // set the owning side to null (unless already changed)
            if ($classFeeSchedule->getInvoiceId() === $this) {
                $classFeeSchedule->setInvoiceId(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection|ClassSessionEnrolmentFeeSchedule[]
     */
    public function getClassSessionEnrolmentFeeSchedules(): Collection
    {
        return $this->classSessionEnrolmentFeeSchedules;
    }

    public function addClassSessionEnrolmentFeeSchedule(ClassSessionEnrolmentFeeSchedule $classSessionEnrolmentFeeSchedule): self
    {
        if (!$this->classSessionEnrolmentFeeSchedules->contains($classSessionEnrolmentFeeSchedule)) {
            $this->classSessionEnrolmentFeeSchedules[] = $classSessionEnrolmentFeeSchedule;
            $classSessionEnrolmentFeeSchedule->setInvoice($this);
        }

        return $this;
    }

    public function removeClassSessionEnrolmentFeeSchedule(ClassSessionEnrolmentFeeSchedule $classSessionEnrolmentFeeSchedule): self
    {
        if ($this->classSessionEnrolmentFeeSchedules->contains($classSessionEnrolmentFeeSchedule)) {
            $this->classSessionEnrolmentFeeSchedules->removeElement($classSessionEnrolmentFeeSchedule);
            // set the owning side to null (unless already changed)
            if ($classSessionEnrolmentFeeSchedule->getInvoice() === $this) {
                $classSessionEnrolmentFeeSchedule->setInvoice(null);
            }
        }

        return $this;
    }

    /*
     * CUSTOM FUNCTIONS :-)
     *
     * */

    ## invoice always contains invoice items, so when we want to calculate the total amount of invoice we will count the invoice item amount.
    ## this function is doing so,.
    public function getTotalAmountOfInvoice()
    {
        $invoiceItems = $this->getInvoiceItems();
        $amountSum = 0;
        foreach ($invoiceItems as $item) {
            $amountSum += $item->getInvItemAmount();
        }
        return $amountSum;
    }

    ## get the invoice status,
    ## is this invoice is paid, un-paid, or partial pay?
    public function getTheInvoiceStatus()
    {
        $transactions = $this->getTransactions();
        $transactionAmount = 0;
        $status = 'UnPaid';
        $totalAmountOfInvoice = $this->getTotalAmountOfInvoice();
        foreach ($transactions as $value) {
            $transactionAmount += $value->getAmount();
        }
        if ($transactionAmount == $totalAmountOfInvoice) {
            ## if invoice has been paid, transaction amount is equal to the invoice items.
            $status = 'Paid';
        } else if ($transactionAmount > 0 && $totalAmountOfInvoice > $transactionAmount) {
            ## if some amount of invoice has been paid,
            $status = 'Partial Paid';
            $this->invoiceAmountWhichIsNotPaid = $totalAmountOfInvoice - $transactionAmount;
        } else if ($transactionAmount > $totalAmountOfInvoice) {
            ## if amount transfer to the account more than the invoice amount.
            $status = 'Over Paid';
            $this->invoiceAmountWhichIsNotPaid = $totalAmountOfInvoice - $transactionAmount;
        } else if ($transactionAmount == 0) {
            $status = 'Un Paid';
            $this->invoiceAmountWhichIsNotPaid = $totalAmountOfInvoice;
        }

        return $status;

    }

    ## get the invoice amount which is not paid.,
    public function getTheInvoiceAmountWhichIsNotPaid()
    {
        return $this->invoiceAmountWhichIsNotPaid;
    }

    public function getClassSessionEnrolment(): ?ClassSessionEnrolment
    {
        return $this->class_session_enrolment;
    }

    public function setClassSessionEnrolment(?ClassSessionEnrolment $class_session_enrolment): self
    {
        $this->class_session_enrolment = $class_session_enrolment;

        return $this;
    }

    public function getIsCancel(): ?bool
    {
        return $this->is_cancel;
    }

    public function setIsCancel(bool $is_cancel = false): self
    {
        $this->is_cancel = $is_cancel;

        return $this;
    }

}
